package spec.benchmarks._239_nih; 
import java.awt.*;
import java.awt.image.*;
import java.net.URL;

public class ImagePlus {

	static final int GRAY8 = 0;
	static final int GRAY16 = 1;
	static final int GRAY32 = 2;
	static final int COLOR_256 = 3;
	static final int COLOR_RGB = 4;
	static final int HSB = 5;
	static final int RGBA = 6;
   	public static Image specIMG = null;
	public static URL lastURL = null; 

	private Image img = null;
	private ImageProcessor ip = null;
	private ImageWindow win = null;
	private ImageJ ij = null;
	private String title;
	private	URL url = null;
	private int width = 0;
	private int height = 0;
	private LookUpTable lut = null;
	private int imageType = GRAY8;
	private Stack stack = null;
	private Roi roi = null;
	private double magnification = 1.0;
	private Rectangle srcRect;
	protected boolean reserved = false;

	int testField = 0;

    public ImagePlus(String title, Image img, ImageJ ij) {
		this.title = title;
		this.ij = ij;
		if (img!=null)
			setImage(img);
    }
    

    public ImagePlus(String title, short[] pixels16, int width, int height, ImageJ ij) {
		this.title = title;
		this.ij = ij;
		setImage(pixels16, width, height, false);
    }
    

    public ImagePlus(String title, int width, int height, ImageJ ij) {
		this.title = title;
		this.width = width;
		this.height = height;
		this.ij = ij;
    }


    public ImagePlus(String title, String urlString, ImageJ ij) {
    	
		this.title = title;
		this.ij = ij;
        try {
        	url = new URL(urlString);
        }
        catch (Exception e) {
            ij.write("URL error: " + urlString);
            url = null;
        }
        if (url != null) {
			ij.showStatus("Loading: " + url);
			if (!(url.equals(lastURL))) {
				specIMG = Toolkit.getDefaultToolkit().getImage(url);
				lastURL = url;
			}
        }
		if (specIMG!=null)
			setImage(specIMG);
		ij.showStatus("");
    }

	public synchronized boolean reserve() {
		if (reserved)
			return false;
        else {
        	reserved = true;
			return true;
        }
	}
	
	
	public synchronized void unreserve() {
		reserved = false;
	}
	
	
    private boolean myWaitForID(int id, MediaTracker tracker) throws InterruptedException {
		boolean first = true;
		double progress = 0.0;
		double inc = 0.03;

		while (true) {
		    int status = tracker.statusID(id, first);
		    if ((status & tracker.LOADING) == 0) {
				Info.showProgress(1.0);
				return (status == tracker.COMPLETE);
			}
		    first = false;
		    Info.showProgress(progress);
		    progress += inc;
		    if (progress>=1.0) {
		    	progress = 0.95;
		    	inc = -0.03;
		    }
		    if (progress<=0) {
		    	progress = 0.03;
		    	inc = 0.03;
		    }
			try {Thread.sleep(100);}
			catch (InterruptedException e) { }
		}
    }


	private void waitForImage(Image img) {
		if (img.getHeight(ij)==-1) {
			MediaTracker tracker = new MediaTracker(ij);
			tracker.addImage(img, 0);
			try {myWaitForID(0, tracker);}
			catch (InterruptedException e){}
		}
	}
	

	void show() {
		
	
		if ((img!=null) && (width>=0) && (height>=0)) {
			win = new ImageWindow(this);
			draw();
			Info.showStatus("Dimensions: " + width + " x " + height);
		
		}
		else
			ij.write("Error loading image");

	}
	
	
	Image getImage() {
		return img;
	}
	

	ImageJ getImageJ() {
		return ij;
	}
	

	void setImage(Image img) {
		int type;
		
	    killProcessor();
	    roi = null;
	    waitForImage(img);
		this.img = img;
		width = img.getWidth(ij);
		height = img.getHeight(ij);
		srcRect = new Rectangle(0, 0, width, height);
		magnification = 1.0;
		lut = new LookUpTable(img, ij);
		if (lut.getMapSize() > 0) {
			if (lut.isGrayscale())
				type = GRAY8;
			else
				type = COLOR_256;
		}
		else {
			type = COLOR_RGB;
		}
		setType(type);
	}
	

	void setImage(short[] pixels16, int width, int height, boolean whiteIsZero) {
	// Insert 16-bit image
		this.width = width;
		this.height = height;
		srcRect = new Rectangle(0, 0, width, height);
		magnification = 1.0;
	    roi = null;
	    stack = null;
	    lut = null;
	    ColorModel cm = Converter.makeGrayscaleColorModel(whiteIsZero);
	    ip = new ShortProcessor(pixels16, width, height, cm, ij.getProgressBar());
		if (Info.debugMode) Info.write(title + ": new ShortProcessor");
	    img = ip.createImage();
		setType(GRAY16);
	}
	

	void setImage(float[] pixels32, int width, int height, boolean whiteIsZero) {
	// Insert 32-bit real image
		this.width = width;
		this.height = height;
		srcRect = new Rectangle(0, 0, width, height);
		magnification = 1.0;
	    roi = null;
	    stack = null;
	    lut = null;
	    ColorModel cm = Converter.makeGrayscaleColorModel(whiteIsZero);
	    ip = new FloatProcessor(pixels32, width, height, cm, ij.getProgressBar());
		if (Info.debugMode) Info.write(title + ": new FloatProcessor");
	    img = ip.createImage();
		setType(GRAY32);
	}
	

	void udateImage(Image img) {
		this.img = img;
	}


	ImageWindow getWindow() {
		return win;
	}
	

	void setWindow(ImageWindow win) {
		this.win = win;
	}
	

	private final void setupProcessor() {
		if (imageType==COLOR_RGB) {
			if (ip == null || ip instanceof ByteProcessor) {
				ip = new ColorProcessor(img, ij.getProgressBar());
				if (Info.debugMode) Info.write(title + ": new ColorProcessor");
			}
		}
		else if (ip==null || (ip instanceof ColorProcessor)) {
			ip = new ByteProcessor(img, ij.getProgressBar());
			if (Info.debugMode) Info.write(title + ": new ByteProcessor");
		}
		if (roi!=null)
			ip.setRoi(roi.getBoundingRect());
	}
	
	
	boolean isProcessor() {
		return ip!=null;
	}
	
	ImageProcessor getProcessor() {
		setupProcessor();
		return ip;
	}
	
	void killProcessor() {
		if ((imageType!=GRAY16) && (imageType!=GRAY32) && magnification==1.0) {
			if (ip!=null && Info.debugMode) Info.write(title + ": killProcessor");
			ip = null;
		}
	}
	

	int[] getMask () {
		if (roi!=null)
			return roi.getMask();
		else
			return null;
	}


	public ImageStatistics getStatistics() {
		ImageProcessor ip = getProcessor();
		ip.setMask(getMask());
		switch (getType()) {
			case GRAY16:
				return new ShortStatistics(ip);
			case GRAY32:
				return new FloatStatistics(ip);
			case COLOR_RGB:
				return new ColorStatistics(ip);
			default:
				return new ByteStatistics(ip);
		}
	}
	

	Graphics getGraphics() {
		return img.getGraphics();
	}

	String getTitle() {
    	return title;
    }

    int getWidth() {
    	return width;
    }

    final int getWidthFinal() {
    	return width;
    }

    int getHeight() {
    	return height;
    }
    

	int getDepth() {
		if (stack==null)
			return 1;
		else
			return stack.getSize();
	}
	

	int offScreenX(int x) {
	// converts a screen x-coordinate to an offscreen x-coordinate
		return srcRect.x + (int)(x/magnification);
	}
	
	int offScreenY(int y) {
	// converts a screen y-coordinate to an offscreen y-coordinate
		return srcRect.y + (int)(y/magnification);
	}
	
	int screenX(int x) {
	// converts an offscreen x-coordinate to a screen x-coordinate
		return  (int)((x-srcRect.x)*magnification);
	}
	
	int screenY(int y) {
	// converts an offscreen y-coordinate to a screen y-coordinate
		return  (int)((y-srcRect.y)*magnification);
	}
	
	
	double getMagnification() {
		return magnification;
	}
	
	
	void zoom (double scale, Event event) {
		double mag = magnification*scale;
		if (mag<=1.0) {
			unzoom();
			return;
		}
		if (mag>64.0)
			return;
		int imag = (int)mag;
		if (imag>1) {
			int w = width/imag;
			int h = height/imag;
			int x = offScreenX(event.x);
			int y = offScreenY(event.y);
			Rectangle r = new Rectangle(x-w/2, y-h/2, w, h);
			if (r.x<0) r.x = 0;
			if (r.y<0) r.y = 0;
			if (r.x+w>width) r.x = width-w;
			if (r.y+h>height) r.y = height-h;
			srcRect = r;
			//Info.write("" + r + " " + x);
		}
		magnification = mag;
		ImageProcessor ip = getProcessor();
		updateAndDraw();
		win.setTitle(getTitle() + " (" + imag + ":1)");
	}
	
	
	void unzoom () {
		magnification = 1.0;
		srcRect = new Rectangle(0, 0, width, height);
		updateAndDraw();
		win.setTitle(getTitle());
	}
	
	
	void drawImage(Graphics g) {
	// draw image at current magnification
		int imag = (int)magnification;
		if (imag==1)
			g.drawImage(img, 0, 0, null);
		else if (imag>1) {
			g.drawImage(img, -srcRect.x*imag, -srcRect.y*imag, width*imag, height*imag, null);
		}
		else {
			g.setColor(Color.white);
			g.fillRect(0, 0, width, height);
			g.drawImage(img, 0, 0, (int)(width*magnification), (int)(height*magnification), null);
		}
	}
	
	
	void draw(){
	// draw image and roi
		if ((win!=null) && img!=null) {
			ImageCanvas ic = win.getCanvas();
			Graphics g = ic.getGraphics();
			if (g!=null) {
				if (Info.debugMode) Info.write(title + ": drawImage");
				g.drawImage(img, 0, 0, null);
				if (roi != null)
					roi.draw(g);
			}
		}
		//if (win!=null)
		//	win.getCanvas().repaint();
	}
	
	
	void draw(int x, int y, int width, int height){
	// draw image and roi using clip rect
		if ((win!=null) && img!=null) {
			width = (int)(width*magnification);
			height = (int)(height*magnification);
			x = screenX(x);
			y = screenY(y);
			ImageCanvas ic = win.getCanvas();
			Graphics g = ic.getGraphics();
			Graphics g2 = g.create();
			g2.clipRect(x, y, width, height);
			g.drawImage(img, 0, 0, null);
			if (roi != null)
				roi.draw(g2);
			g2.dispose();
			g2 = null;

		}
		//if (win!=null)
		//	win.getCanvas().repaint(x, y, width, height);
	}
	
	
	void drawRoi(Graphics g) {
		if (roi!=null)
			roi.draw(g);
	}
	
	
	void updateAndDraw() {
		if (ip == null)
			return;
		//if (img!=null) img.flush();
		if (magnification!=1.0)
			img = ip.createImage(srcRect, width, height);
		else
			img = ip.createImage();
		if (img!=null) {
			if (roi!=null)
				draw(roi.x, roi.y, roi.width+1, roi.height+1);
			else
				draw();
		}
		else
			Info.outOfMemory("updateAndDraw");
	}
	
	
    int getType() {
    	return imageType;
    }


    void setType(int type) {
    	if ((type<0) || (type>RGBA))
    		return;
    	imageType = type;
    	switch (imageType) {
	    	case GRAY8:
				killStack();
				break;
	    	case RGBA:
				break;
	    	case HSB:
				break;
	    	case COLOR_RGB:
			    lut = null;
				killStack();
			    break;
			case COLOR_256:
				killStack();
				break;
	    }
		if (win!=null)
			Menus.updateMenus();
    }


    LookUpTable getLUT() {
    	if (lut == null)
    		lut = new LookUpTable(img, ij);
    	return lut;
    }
    
	
	float[] getPixel(int x, int y) {
		float[] value = {0F, 0F, 0F, 0F};
		if (img == null)
			return value;
		switch (imageType) {
			case GRAY8:
			case COLOR_256:
				int index;
				if (ip!=null)
					index = (int)ip.getPixel(x, y);
				else {
					byte[] pixels8 = new byte[1];
					BytePixelGrabber bpg = new BytePixelGrabber(img, x, y, 1, 1, pixels8, 0, width);
					try {bpg.grabPixels();}
					catch (InterruptedException e){return value;};
					index = pixels8[0] & 0xff;
				}
				value[0] = index;
				if ((lut != null) && (lut.getMapSize()>=index)) {
					byte[] colors = lut.getColors();
					value[1] = colors[index*3+0] & 0xff;
					value[2] = colors[index*3+1] & 0xff;
					value[3] = colors[index*3+2] & 0xff;
				}
				break;
			case GRAY16:
			case GRAY32:
				value[0] = ip.getPixel(x, y);
				break;
			default:
				int[] pixels32 = new int[1];
				PixelGrabber pg = new PixelGrabber(img, screenX(x), screenY(y), 1, 1, pixels32, 0, width);
				try {pg.grabPixels();}
				catch (InterruptedException e){return value;};
				int c = pixels32[0];
				int r = (c&0xff0000)>>16;
				int g = (c&0xff00)>>8;
				int b = c&0xff;
				value[0] = r;
				value[1] = g;
				value[2] = b;
				break;
		}
		return value;
	}
    
	
	Stack getStack() {
		if (stack==null)
			stack = new Stack(this);
		return stack;
	}
	

	void killStack() {
		stack = null;
		killProcessor();
	}
			

	void undoFilter() {
		if (ip!=null) {
			ip.reset();
			updateAndDraw();
			if (Info.debugMode) Info.write(getTitle() + ": undo filter");
		}
	}

	public Roi getRoi() {
		return roi;
	}
	

   void killRoi() {
		if (roi!=null) {
			roi = null;
			if (ip!=null)
				ip.setRoi(null);
			draw();
		}
	}
	

	void handleRoiMouseDown(Event event) {
		int x = offScreenX(event.x);
		int y = offScreenY(event.y);
		if (roi!=null) {
			if (roi.contains(x, y)) {
				roi.handleMouseDown(x, y);
				return;
			}
			if (roi.getType()==roi.POLYGON && roi.state==roi.CONSTRUCTING)
				return;
		}
		// Create new roi
		killRoi();
		switch (Toolbar.getToolId()) {
			case Toolbar.RECTANGLE:
				roi = new Roi(x, y, this);
				break;
			case Toolbar.OVAL:
				roi = new OvalRoi(x, y, this);
				break;
			case Toolbar.POLYGON:
				roi = new PolygonRoi(x, y, this, Toolbar.POLYGON);
				break;
			case Toolbar.FREEHAND:
				roi = new FreehandRoi(x, y, this);
				break;
		}
	}


	void handleMouseDown(Event event) {
		switch (Toolbar.getToolId()) {
			case Toolbar.MAGNIFYING_GLASS:
				if ((event.modifiers & (Event.ALT_MASK|Event.META_MASK|Event.CTRL_MASK))!=0)
					zoom(0.5, event);
				else
					zoom(2.0, event);
				break;
			default:  //selection tool
				handleRoiMouseDown(event);
		}
	}
	
	
	void handleMouseUp(Event event) {
		if (roi != null) {
			if ((roi.width==0 || roi.height==0) && !(roi.getType()==roi.POLYGON  && roi.state==roi.CONSTRUCTING))
				killRoi();
			else
				roi.handleMouseUp(event.x, event.y);
		}
	}


	void handleMouseDrag(Event event) {
		if (roi != null)
			roi.handleMouseDrag(offScreenX(event.x), offScreenY(event.y), (event.modifiers&Event.SHIFT_MASK)!=0);
	}
	
	
	void handleMouseMove(int x, int y) {
	// x and y are offscreen coordinates
		if (roi!=null && roi.getType()==roi.POLYGON && roi.state==roi.CONSTRUCTING) {
				PolygonRoi pRoi = (PolygonRoi)roi;
				pRoi.handleMouseMove(x, y);
		}
		else
			Info.showStatus("Location = (" + x + "," + y + ")" + showValue(x, y));
	}
	
	

    private String showValue(int x, int y) {
    	float[] value = getPixel(x, y);
    	if (getType()==GRAY8 || getType()==GRAY16 || getType()==GRAY32)
    		return(", value=" + value[0]);
    	else if (getType()==COLOR_256)
    		return(", index=" + value[0] + ", value=" + value[1] + "," + value[2] + "," + value[3]);
    	else
    		return(", value=" + value[0] + "," + value[1] + "," + value[2]);
    }
    
    
	URL getURL() {
		return url;
	}
	
	
	void revert() {
    	if (url!=null) {
			killStack();
			unzoom();
			Info.showStatus("Loading: " + url);
	        img = Toolkit.getDefaultToolkit().getImage(url);
	        setImage(img);
			if (win!=null)
				win.getCanvas().repaint();
			Info.showStatus("");
		}
    }
    
    
    FileInfo getInfo() {
    	FileInfo fi = new FileInfo();
    	fi.width = width;
    	fi.height = height;
    	fi.blackIsZero = !getLUT().isInverted();
    	setupProcessor();
    	fi.pixels = ip.getPixels();
    	fi.ij = ij;
    	switch (imageType) {
	    	case GRAY8:
				fi.fileType = fi.GRAY8;
				break;
	    	case GRAY16:
				fi.fileType = fi.GRAY16;
				break;
	    	case GRAY32:
				fi.fileType = fi.GRAY32_FLOAT;
				break;
			case COLOR_256:
				fi.fileType = fi.COLOR8;
				fi.colorTable = new byte[768];
				byte[] colors = lut.getColors();
				for (int i=0; i<lut.getMapSize(); i++) {
					fi.colorTable[i] = colors[3*i];
					fi.colorTable[256+i] = colors[3*i+1];
					fi.colorTable[512+i] = colors[3*i+2];
				}
				break;
	    	case COLOR_RGB:
				fi.fileType = fi.RGB;
				break;
			default:
    	}
    	return fi;
    }

}


