package spec.benchmarks._239_nih; 
// Processes RGB images. Based on the ImageProcessor class from "KickAss Java
// Programming" by Tonny Espeset (http://www.sn.no/~espeset)

import java.util.*;
import java.awt.*;
import java.awt.image.*;
import spec.harness.Context; 

class ColorProcessor extends ImageProcessor {

	private int[] pixels;
	private int[] snapshotPixels = null;
	private int fgColor = 0xff000000; //black
	private int bgColor = 0xffffffff; //white


	ColorProcessor(Image img) {
		width = img.getWidth(null);
		height = img.getHeight(null);
		pixels = new int[width * height];
		PixelGrabber pg = new PixelGrabber(img, 0, 0, width, height, pixels, 0, width);
		try {
			pg.grabPixels();
		} catch (InterruptedException e){};
		setRoi(null);
	}


	ColorProcessor(Image img, ProgressBar bar) {
		this(img);
		progressBar = bar;
	}


	ColorProcessor(int width, int height, byte[] R, byte[] G, byte[] B, byte[] A) {
	// creates a ColorProcessor from red, green, blue and alpha byte arrays
		this.width = width;
		this.height = height;
		pixels = new int[width * height];
		setRGBA(R, G, B, A, 0.0, 0.0);
		setRoi(null);
	}


	ColorProcessor(int width, int height, byte[] H, byte[] S, byte[] B) {
	// creates a ColorProcessor from hue, saturation and brightness byte arrays
		this.width = width;
		this.height = height;
		pixels = new int[width * height];
		setHSB(H, S, B, 0.0, 0.0);
		setRoi(null);
	}
	

	Image createImage() {
		Image img = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, pixels, 0, width));
		return img;
	}


	Image createImage(Rectangle srcRect, int dstWidth, int dstHeight) {
	// zoom using pixel replication
		int imag = dstWidth/srcRect.width;
		int[] pixels2 = new int[width*height];
		int i = 0;
		for (int y=0; y<height; y++) {
			int j = (srcRect.y + y/imag)*width + srcRect.x;
			for (int x=0; x<width; x++)
				pixels2[i++] = pixels[j+x/imag];
		}	    
		return Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, pixels2, 0, width));
	}


	Color getColor(int x, int y) {
		int c = pixels[y*width+x];
		int r = (c&0xff0000)>>16;
		int g = (c&0xff00)>>8;
		int b = c&0xff;
		return new Color(r,g,b);
	}


	void snapshot() {
	//Make a snapshot of the current image
		snapshotWidth = width;
		snapshotHeight = height;
		if (snapshotPixels==null || (snapshotPixels!=null && snapshotPixels.length!=pixels.length))
			snapshotPixels = new int[width * height];
		System.arraycopy(pixels, 0, snapshotPixels, 0, width*height);
		newSnapshot = true;
	}


	void reset() {
	//Reset the image from snapshot
		if (snapshotPixels!=null) {
				if (width!=snapshotWidth || height!=snapshotHeight) {
				width = snapshotWidth;
				height = snapshotHeight;
				pixels = new int[width * height];
			}
			System.arraycopy(snapshotPixels, 0, pixels, 0, width*height);
			newSnapshot = true;
		}
	}


	void reset(int[] mask) {
	// Restore pixels that are within roi but not part of mask
		for (int y=roiY, my=0; y<(roiY+roiHeight); y++, my++) {
			int i = y * width + roiX;
			int mi = my * roiWidth;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				if (mask[mi++]==-1)
					pixels[i] = snapshotPixels[i];
				i++;
			}
		}
	}


	int[] getPixelsCopy() {
		if (newSnapshot)
			return snapshotPixels;
		else {
			int[] pixels2 = new int[width*height];
        	System.arraycopy(pixels, 0, pixels2, 0, width*height);
			return pixels2;
		}
	}


	float getPixel(int x, int y) {
		if (x>=0 && x<width && y>=0 && y<height) {
			int c = pixels[y*width+x];
			int r = (c&0xff0000)>>16;
			int g = (c&0xff00)>>8;
			int b = c&0xff;
			return (float)(r*0.30 + g*0.59 + b*0.11);
		}
		else
			return 0f;
	}

	float getPixel(int pixel) {
		int c = pixel;
		int r = (c&0xff0000)>>16;
		int g = (c&0xff00)>>8;
		int b = c&0xff;
		return (float)(r*0.30 + g*0.59 + b*0.11);
	}

	Object getPixels() {
		return (Object)pixels;
	}


	void getHSB(byte[] H, byte[] S, byte[] B, double pStart, double pWidth) {
	// returns hue, saturation and brightness in 3 byte arrays
		int c, r, g, b;
		float[] hsb = new float[3];
		for (int i=0; i < width*height; i++) {
			c = pixels[i];
			r = (c&0xff0000)>>16;
			g = (c&0xff00)>>8;
			b = c&0xff;
			hsb = Color.RGBtoHSB(r, g, b, hsb);
			H[i] = (byte)((int)(hsb[0]*255.0) & 0xff);
			S[i] = (byte)((int)(hsb[1]*255.0) & 0xff);
			B[i] = (byte)((int)(hsb[2]*255.0) & 0xff);
			if (i%10000==0 && pWidth>0.0)
				showProgress(pStart + pWidth*((double)i/(width*height)));
		}
	}
	

	void getRGBA(byte[] R, byte[] G, byte[] B, byte[] A, double pStart, double pWidth) {
	// returns red, green, blue and alpha in 4 byte arrays
		int c, a, r, g, b;
		for (int i=0; i < width*height; i++) {
			c = pixels[i];
			a = (c&0xff000000)>>>24;
			r = (c&0xff0000)>>16;
			g = (c&0xff00)>>8;
			b = c&0xff;
			R[i] = (byte)(r&0xff);
			G[i] = (byte)(g&0xff);
			B[i] = (byte)(b&0xff);
			A[i] = (byte)(a&0xff);
			if (i%10000==0 && pWidth>0.0)
				showProgress(pStart + pWidth*((double)i/(width*height)));
		}
	}


	void setRGBA(byte[] R, byte[] G, byte[] B, byte[] A, double pStart, double pWidth) {
	// sets the current pixels from 4 byte arrays (reg, green, blue and alpha)
		int c, a, r, g, b;
		for (int i=0; i < width*height; i++) {
			pixels[i] = ((A[i]&0xff)<<24) | ((R[i]&0xff)<<16) | ((G[i]&0xff)<<8) | B[i]&0xff;
			if (i%10000==0 && pWidth>0.0)
				showProgress(pStart + pWidth*((double)i/(width*height)));
		}
	}


	void setHSB(byte[] H, byte[] S, byte[] B, double pStart, double pWidth) {
	// sets the current pixels from 3 byte arrays (hue, saturation and brightness)
		float hue, saturation, brightness;
		for (int i=0; i < width*height; i++) {
			hue = (float)((H[i]&0xff)/255.0);
			saturation = (float)((S[i]&0xff)/255.0);
			brightness = (float)((B[i]&0xff)/255.0);
			pixels[i] = Color.HSBtoRGB(hue, saturation, brightness);
			if (i%10000==0 && pWidth>0.0)
				showProgress(pStart + pWidth*((double)i/(width*height)));
		}
	}
	

	void insert(ImageProcessor ip, int xloc, int yloc) {
	//inserts the image in ip at (xloc, yloc)
		Rectangle r1, r2;
		int srcIndex, dstIndex;
		int srcWidth, srcHeight;
		int xSrcBase, ySrcBase;
		int[] srcPixels;
		
		srcWidth = ip.getWidth();
		srcHeight = ip.getHeight();
		r1 = new Rectangle(srcWidth, srcHeight);
		r1.move(xloc, yloc);
		r2 = new Rectangle(width, height);
		if (!r1.intersects(r2))
			return;
		srcPixels = (int[])ip.getPixels();
		r1 = r1.intersection(r2);
		xSrcBase = (xloc<0)?-xloc:0;
		ySrcBase = (yloc<0)?-yloc:0;
		for (int y=r1.y; y<(r1.y+r1.height); y++) {
			srcIndex = (y-yloc)*srcWidth + (r1.x-xloc);
			dstIndex = y * width + r1.x;
			for (int x=r1.x; x<(r1.x+r1.width); x++)
				pixels[dstIndex++] = srcPixels[srcIndex++];
		}
		newSnapshot = false;
	}
	
	
	/* Filters start here */

	void applyTable(byte[] lut) {
		for (int y=roiY; y<(roiY+roiHeight); y++) {
			int i = y * width + roiX;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				int c = pixels[i];
				int r = lut[(c&0xff0000)>>16]&0xff;
				int g = lut[(c&0xff00)>>8]&0xff;
				int b = lut[c&0xff]&0xff;
				pixels[i] = (c&0xff000000) + (r<<16) + (g<<8) + b;
				i++;
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		hideProgress();
	}


    void noise(double range) {
		Random rnd=new Random();
		int c, r, g, b;

		for (int y=roiY; y<(roiY+roiHeight); y++) {
			int i = y * width + roiX;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				int RandomBrightness = (int)(rnd.nextGaussian()*range);

				c=pixels[i];
				r=(c&0xff0000)>>16;
				g=(c&0xff00)>>8;
				b=c&0xff;
				
				g += RandomBrightness;
				if (g < 0)
					g = 0;
				if (g > 255)
					g = 255;

				b += RandomBrightness;
				if (b < 0)
					b = 0;
				if (b > 255)
					b = 255;

				pixels[i]=(c&0xff000000)+(r<<16)+(g<<8)+b;
				i++;
			}
			if (y%10==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		hideProgress();
    }


	public void crop() {
		int[] pixels2 = getPixelsCopy();
		for (int ys=roiY; ys<roiY+roiHeight; ys++) {
			int srcoff = roiX+ys*width;
			System.arraycopy(pixels2, roiX+ys*width, pixels, (ys-roiY)*roiWidth, roiWidth);
		}
		width=roiWidth;
		height=roiHeight;
		roiX = 0;
		roiY = 0;
		newSnapshot = false;
	}
	

	/*
	public void move(int dx, int dy)
	{
	  //Duplicate image
	  int[] sourcePixels=pixels;
	  pixels = new int[width * height];

	  int index=0;
	  for (int y=0;y<height;y++)
		 for (int x=0;x<width;x++)
		 {
			int xs=x-dx;
			int ys=y-dy;
			if ((xs>=0)&&(xs<width)&&(ys>=0)&&(ys<height))
			  pixels[index++]=sourcePixels[width*ys+xs];
			else
			  pixels[index++]=bgColor;
		 }
	}
	*/


	void scale(double xScale, double yScale, boolean resizeImage) {
		int dstWidth, dstHeight;
        int xMin, xMax, yMin, yMax;
        int srcCenterX, srcCenterY, dstCenterX, dstCenterY;
		ColorModel defaultCM = ColorModel.getRGBdefault(); 

		srcCenterX = roiX + roiWidth/2;
		srcCenterY = roiY + roiHeight/2;
		if (resizeImage) {
			dstWidth = (int)(roiWidth*xScale);
			dstHeight = (int)(roiHeight*yScale);
			dstCenterX = dstWidth/2;
			dstCenterY = dstHeight/2;
	        xMin = 0;
			xMax = dstWidth - 1;
	        yMin = 0;
	        yMax = dstHeight - 1;
		}
		else {
			dstWidth = width;
			dstHeight = height;
			dstCenterX = srcCenterX;
			dstCenterY = srcCenterY;
			if ((xScale>1.0) && (yScale>1.0)) {
				//expand roi
		        xMin = dstCenterX - (int)((dstCenterX-roiX)*xScale);
		        if (xMin<0)
		        	xMin = 0;
		        xMax = xMin + (int)(roiWidth*xScale) - 1;
		        if (xMax>=width)
		        	xMax = width - 1;
		        yMin = dstCenterY - (int)((dstCenterY-roiY)*yScale);
		        if (yMin<0)
		        	yMin = 0;
		        yMax = yMin + (int)(roiHeight*yScale) - 1;
		        if (yMax>=height)
		        	yMax = height - 1;
			}
			else {
		        xMin = roiX;
				xMax = roiX + roiWidth - 1;
		        yMin = roiY;
		        yMax = roiY + roiHeight - 1;
	        }
        }
        long checksum = 0; 
       	int[] pixels2 = getPixelsCopy();
        if (resizeImage)
        	pixels = new int[dstWidth * dstHeight];
        boolean checkCoordinates = !resizeImage && ((xScale < 1.0) || (yScale < 1.0));
        for (int y=yMin; y<=yMax; y++) {
            int index = y*dstWidth + xMin;
			int ys = (int)((y-dstCenterY)/yScale) + srcCenterY;
            for (int x=xMin; x<=xMax; x++) {
				int xs = (int)((x-dstCenterX)/xScale) + srcCenterX;
                if (checkCoordinates && ((xs<xMin) || (xs>xMax) || (ys<yMin) || (ys>yMax)))
                	  pixels[index++] = bgColor;
                else
				{
					  pixels[index++] = pixels2[width*ys+xs];
				}
			}
   			checksum += defaultCM.getGreen(pixels[index -1]);  
			if (y%20==0)
				showProgress((double)(y-yMin)/dstHeight);
        }
		spec.harness.Context.out.println("Scaling Checksum = " + checksum%10000);
        width = dstWidth;
        height = dstHeight;
        if (resizeImage)
        	setRoi(null);
		hideProgress();
	}


	void rotate(double angle) {
        if (angle%360==0)
        	return;
       	int[] pixels2 = getPixelsCopy();
		int centerX = roiX + roiWidth/2;
		int centerY = roiY + roiHeight/2;
		int width = this.width;  //Local variables are faster than instance variables
        int height = this.height;
        int roiX = this.roiX;
        int xMax = roiX + this.roiWidth - 1;
        int SCALE = 1024;
        long checksum = 0;
		ColorModel defaultCM = ColorModel.getRGBdefault(); 
        //Convert from degrees to radians and calculate cos and sin of angle
        //Negate the angle to make sure the rotation is clockwise
        double angleRadians = -angle/(180.0/Math.PI);
        int ca = (int)(Math.cos(angleRadians)*SCALE);
        int sa = (int)(Math.sin(angleRadians)*SCALE);
        int temp1 = centerY*sa - centerX*ca;
        int temp2 = -centerX*sa - centerY*ca;
        
        for (int y=roiY; y<(roiY + roiHeight); y++) {
            int index = y*width + roiX;
            int temp3 = (temp1 - y*sa)/SCALE + centerX;
            int temp4 = (temp2 + y*ca)/SCALE + centerY;
            for (int x=roiX; x<=xMax; x++) {
                //int xs = (int)((x-centerX)*ca-(y-centerY)*sa)+centerX;
                //int ys = (int)((y-centerY)*ca+(x-centerX)*sa)+centerY;
                int xs = (x*ca)/SCALE + temp3;
                int ys = (x*sa)/SCALE + temp4;
                if ((xs>=0) && (xs<width) && (ys>=0) && (ys<height))
                	  pixels[index++] = pixels2[width*ys+xs];
                else
                	  pixels[index++] = bgColor;
            }
			checksum += defaultCM.getGreen(pixels[index -1]); 
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
        }
		spec.harness.Context.out.println("Rotation  Checksum = " + checksum%10000);
		hideProgress();
	}


	void flipVertical() {
		int[] pixels2 = getPixelsCopy();
		int lastLine=width*(roiY + roiHeight) + roiX - width;
		for (int offset=width*roiY+roiX; offset<(width*(roiY + roiHeight)); offset+=width) {
			System.arraycopy(pixels2, offset, pixels, lastLine, roiWidth);
			lastLine -= width;
		}
		newSnapshot = false;
	}


	void flipHorizontal() {
		long checksum = 0;
		ColorModel defaultCM = ColorModel.getRGBdefault(); 
		int[] pixels2 = getPixelsCopy();
		int temp = 2*roiX+roiWidth-1;
		for (int y=roiY; y<(roiY + roiHeight); y++) {
			int yOffset = y*width;
			for (int x=roiX; x<(roiX + roiWidth); x++) {
				pixels[x+yOffset] = pixels2[(temp-x)+yOffset];
				checksum += defaultCM.getGreen(pixels[x+yOffset]); 
			}
		}
		spec.harness.Context.out.println("Flip Horizontal Checksum = " + checksum%10000);
		newSnapshot = false;
	}


	void smooth() {
		int[] pixels2;
		int i, x, y, offset, rowOffset;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
		int rsum = 0, gsum = 0, bsum = 0;
	    long checksum = 0; 
		ColorModel defaultCM = ColorModel.getRGBdefault(); 

		pixels2 = getPixelsCopy();
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = pixels2[offset - width - 2];
			p3 = pixels2[offset - width - 1];
			p5 = pixels2[offset] - 2;
			p6 = pixels2[offset - 1];
			p8 = pixels2[offset + width - 2];
			p9 = pixels2[offset + width - 1];
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = pixels2[offset - rowOffset + 1];
				p4 = p5;
				p5 = p6;
				p6 = pixels2[offset + 1];
				p7 = p8;
				p8 = p9;
				p9 = pixels2[offset + rowOffset + 1];
				
				rsum = (p1 & 0xff0000) + (p2 & 0xff0000) + (p3 & 0xff0000) + (p4 & 0xff0000) + (p5 & 0xff0000)
					+ (p6 & 0xff0000) + (p7 & 0xff0000) + (p8 & 0xff0000) + (p9 & 0xff0000);
				gsum = (p1 & 0xff00) + (p2 & 0xff00) + (p3 & 0xff00) + (p4 & 0xff00) + (p5 & 0xff00)
					+ (p6 & 0xff00) + (p7 & 0xff00) + (p8 & 0xff00) + (p9 & 0xff00);
				bsum = (p1 & 0xff) + (p2 & 0xff) + (p3 & 0xff) + (p4 & 0xff) + (p5 & 0xff)
					+ (p6 & 0xff) + (p7 & 0xff) + (p8 & 0xff) + (p9 & 0xff);
				pixels[offset++] = 0xff000000 | ((rsum/9) & 0xff0000) | ((gsum/9) & 0xff00) | (bsum/9);
			}
			if ((offset % 10) == 0) {
				checksum += defaultCM.getGreen(pixels[offset -1]); 
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		spec.harness.Context.out.println("Smooth Filter Checksum = " + checksum%10000);
		hideProgress();
	}
	

	void sharpen() {
		int[] pixels2;
		int i, x, y, offset, rowOffset;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
		int rsum = 0, gsum = 0, bsum = 0;
	    long checksum = 0; 
		ColorModel defaultCM = ColorModel.getRGBdefault(); 

		pixels2 = getPixelsCopy();
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = pixels2[offset - width - 2];
			p3 = pixels2[offset - width - 1];
			p5 = pixels2[offset] - 2;
			p6 = pixels2[offset - 1];
			p8 = pixels2[offset + width - 2];
			p9 = pixels2[offset + width - 1];
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = pixels2[offset - rowOffset + 1];
				p4 = p5;
				p5 = p6;
				p6 = pixels2[offset + 1];
				p7 = p8;
				p8 = p9;
				p9 = pixels2[offset + rowOffset + 1];
				
				rsum = ((p5 & 0xff0000) >> 16) * 12 - ((p1 & 0xff0000) >> 16) - ((p2 & 0xff0000) >> 16)
					 - ((p3 & 0xff0000) >> 16) - ((p4 & 0xff0000) >> 16) - ((p6 & 0xff0000) >> 16)
					 - ((p7 & 0xff0000) >> 16) - ((p8 & 0xff0000) >> 16) - ((p9 & 0xff0000) >> 16);
				rsum /= 4;
				if (rsum < 0) rsum = 0;
				if (rsum > 255) rsum = 255;
				gsum =  ((p5 & 0xff00) >> 8) * 12 - ((p1 & 0xff00) >> 8) - ((p2 & 0xff00) >> 8)
					- ((p3 & 0xff00) >> 8) - ((p4 & 0xff00) >> 8) - ((p6 & 0xff00) >> 8)
					- ((p7 & 0xff00) >> 8) - ((p8 & 0xff00) >> 8) - ((p9 & 0xff00) >> 8);
				gsum /= 4;
				if (gsum < 0) gsum = 0;
				if (gsum > 255) gsum = 255;
				bsum =  ((p5 & 0xff) * 12) - (p1 & 0xff) - (p2 & 0xff) - (p3 & 0xff) - (p4 & 0xff)
					 - (p6 & 0xff) - (p7 & 0xff) - (p8 & 0xff) - (p9 & 0xff);
				bsum /= 4;
				if (bsum < 0) bsum = 0;
				if (bsum > 255) bsum = 255;
				pixels[offset++] = 0xff000000 | ((rsum << 16) & 0xff0000) | ((gsum << 8 ) & 0xff00) | bsum;
			}
			if ((offset % 10) == 0) {
				checksum += defaultCM.getGreen(pixels[offset -1]); 
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		spec.harness.Context.out.println("Sharpen Filter Checksum = " + checksum%10000);
		hideProgress();
	}
	

	private int sqrt(int x) {
	//see ftp://codon.nih.gov/pub/nih-image/documents/square_root_code.txt
        int r, nr, m;
 
        r = 0;
        m = 0x40000000;
        do {
            nr = r + m;
            if (nr <= x) {
                    x -= nr;
                    r = nr + m;
            }
            r >>= 1;
            m >>= 2;
        } while (m != 0);
        if (x > r)
        	r++;
        return(r);
	}


	void findEdges() {
		byte[] hue = new byte[width * height];
		byte[] saturation = new byte[width * height];
		byte[] brightness = new byte[width * height];
		getHSB(hue, saturation, brightness, 0.0, 0.4);
		ByteProcessor bp = new ByteProcessor(width, height, brightness);
		bp.setRoi(new Rectangle(roiX, roiY, roiWidth, roiHeight));
		bp.findEdges();
		Info.showProgress(.6);
		bp.invert();
		Info.showProgress(.7);
		brightness = (byte[])bp.getPixels();
		setHSB(hue, saturation, brightness, 0.7, 0.3);
		hideProgress();
	}		
		

	void autoThreshold() {
		byte[] R = new byte[width*height];
		byte[] G = new byte[width*height];;
		byte[] B = new byte[width*height];
		byte[] A = new byte[width*height];
		getRGBA(R, G, B, A, 0.0, 0.4);
		Rectangle roi = new Rectangle(roiX, roiY, roiWidth, roiHeight);
		
		ByteProcessor bp = new ByteProcessor(width, height, R);
		bp.setRoi(roi);
		bp.autoThreshold();
		R = (byte[])bp.getPixels();
		Info.showProgress(.5);
		
		bp = new ByteProcessor(width, height, G);
		bp.setRoi(roi);
		bp.setRoi(new Rectangle(roiX, roiY, roiWidth, roiHeight));
		bp.autoThreshold();
		G = (byte[])bp.getPixels();
		Info.showProgress(.6);
		
		bp = new ByteProcessor(width, height, B);
		bp.setRoi(roi);
		bp.autoThreshold();
		B = (byte[])bp.getPixels();
		Info.showProgress(.7);
		
		setRGBA(R, B, G, A, 0.7, 0.3);
		hideProgress();
	}
	
	
	void medianFilter() {
		byte[] hue = new byte[width * height];
		byte[] saturation = new byte[width * height];
		byte[] brightness = new byte[width * height];
		getHSB(hue, saturation, brightness, 0.0, 0.3);
		ByteProcessor bp = new ByteProcessor(width, height, brightness);
		bp.setRoi(new Rectangle(roiX, roiY, roiWidth, roiHeight));
		bp.medianFilter();
		Info.showProgress(.7);
		brightness = (byte[])bp.getPixels();
		setHSB(hue, saturation, brightness, 0.7, 0.3);
		hideProgress();
	}
	
	
	int[] getSpecialHistogram() {
	// special histogram routine only used by Enhance Contrast
		if (mask!=null)
			return getSpecialHistogram(mask);
		int c, r, g, b, v;
		int[] histogram = new int[256];
		for (int y=roiY; y<(roiY+roiHeight); y++) {
			int i = y * width + roiX;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				c = pixels[i++];
				r = (c&0xff0000)>>16;
				g = (c&0xff00)>>8;
				b = c&0xff;
				v = (int)(r*0.30 + g*0.59 + b*0.11);
				histogram[r]++;
				histogram[g]++;
				histogram[b]++;
			}
		}
		return histogram;
	}


	int[] getSpecialHistogram(int[] mask) {
		int c, r, g, b, v;
		int[] histogram = new int[256];
		for (int y=roiY, my=0; y<(roiY+roiHeight); y++, my++) {
			int i = y * width + roiX;
			int mi = my * roiWidth;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				if (mask[mi++]!=-1) {
					c = pixels[i];
					r = (c&0xff0000)>>16;
					g = (c&0xff00)>>8;
					b = c&0xff;
					histogram[r]++;
					histogram[g]++;
					histogram[b]++;
				}
				i++;
			}
		}
		return histogram;
	}


	void enhanceContrast() {
	// histogram stretch
		int[] histogram = getSpecialHistogram();
		int sum = 0;
		for (int i=0; i<255; i++)
			sum += histogram[i];
		int threshold = sum/5000;
		int i = -1;
		boolean found = false;
		do {
			i++;
			found = histogram[i] > threshold;
		} while (!found && i<255);
		int min = i;
		i = 256;
		do {
			i--;
			found = histogram[i] > threshold;
		} while (!found && i>0);
		int max = i;
		if (max>min && (min>0 || max-min<256)) {
			setRoi(null);
			add(-min);
			multiply(256.0/(max-min));
		}
	}


}

