/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.ow2.dsrg.fm.tbplib.TBPResolvingException;
import org.ow2.dsrg.fm.tbplib.TBPUndefinedEmitException;
import org.ow2.dsrg.fm.tbplib.parsed.MethodCall;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedMethodDeclaration;
import org.ow2.dsrg.fm.tbplib.resolved.CVRef;
import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.FPRef;
import org.ow2.dsrg.fm.tbplib.resolved.Reference;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedVardef;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * @author caitt3am
 * 
 */
public class UtilClass {

	/**
	 * Tries to resolve given string as a reference to variable, formal
	 * parameter or constant.
	 * 
	 * @param name
	 *            - What to resolve
	 * @param types
	 *            - collection of accessible types
	 * @param cvars
	 *            - collection of accessible variables, can be null
	 * @param ms
	 *            - type of method where reference occurs, can be null
	 * @return {@link Reference}
	 */
	public static Reference makeReference(String name, List<Typedef> types,
			Map<String, TBPResolvedVardef> cvars, MethodSignature ms) {

		if (ms != null && ms.getParamNames().contains(name)) {
			return new FPRef(name, ms.getParamTypes().get(name));
		} else if (cvars != null && cvars.containsKey(name)) {
			return new CVRef(cvars.get(name));
		} else {
			for (Typedef typedef : types) {
				for (String en : typedef.getEnums()) {
					if (en.equals(name)) {
						return new ConstantRef(name, typedef);
					}
				}
			}
		}
		throw new TBPResolvingException("Cannot resolve this reference: "
				+ name);
	}

	public static CVRef makeCVReference(String name,
			Map<String, TBPResolvedVardef> cvars) {
		if (cvars.containsKey(name))
			return new CVRef(cvars.get(name));
		throw new TBPResolvingException("Cannot resolve this reference: "
				+ name);
	}

	public static ConstantRef makeConstantReference(String name,
			List<Typedef> types) {
		for (Typedef typedef : types) {
			for (String en : typedef.getEnums()) {
				if (en.equals(name)) {
					return new ConstantRef(name, typedef);
				}
			}
		}
		throw new TBPResolvingException("Cannot resolve this reference to constant: "
				+ name);
	}

	/**
	 * Converts list of method decls into map mapping functions fullnames to
	 * signatures with no binding.
	 * 
	 * @param mcs
	 *            method decls
	 * @return method signatures
	 */
	public static Map<String, MethodSignature> generateMethodSignatures(
			List<TBPParsedImperativeNode> mcs, List<Typedef> types) {

		Map<String, MethodSignature> result = new HashMap<String, MethodSignature>();

		for (TBPParsedImperativeNode decl : mcs) {
			MethodSignature signature = generateMS(decl, types);
			result.put(signature.getFullname(), signature);
		}

		return result;
	}

	// Converts TBPParsedMethodDeclaration to MethodSignature
	private static MethodSignature generateMS(TBPParsedImperativeNode node,
			List<Typedef> types) {

		TBPParsedMethodDeclaration decl = (TBPParsedMethodDeclaration) node;

		int s = decl.getTypesAndNames().size();

		assert s % 2 == 0;

		List<String> param_names = new ArrayList<String>(s / 2);
		Map<String, Typedef> param_types = new HashMap<String, Typedef>(s / 2);

		Iterator<String> it = decl.getTypesAndNames().iterator();
		while (it.hasNext()) {
			String s_type = it.next();
			String p_name = it.next();

			param_names.add(p_name);

			for (Typedef t : types) {
				if (t.getName().equals(s_type)) {
					param_types.put(p_name, t);
					break;
				}
			}
			if (!param_types.containsKey(p_name)) {
				throw new TBPResolvingException("Unknown type: " + s_type);
			}

		}
		
		Typedef return_type = null;
		String s_return = decl.getReturnType();
		
		if(s_return != null){
			for(Typedef t : types){
				if(t.getName().equals(s_return)){
					return_type = t;
					break;
				}
			}
			if(return_type == null)
				throw new TBPResolvingException("Unknown type:" + s_return);
		}
		
		return new MethodSignature(decl.getInterface(), decl.getMethod(),
				param_names, param_types, return_type);
	}

	/**
	 * Returns binding actual values filled according to mc. In case of
	 * type mismatch throws exception.
	 * 
	 * @param sigs
	 *            - map of declared methods (must be unbound method signatures)
	 * @param mc
	 *            - actual parameters as strings
	 * @param types
	 *            - collection of accessible types
	 * @param cvars
	 *            - collection of accessible variables, can be null
	 * @param surroundingMethod
	 *            type of method where mc occurs, can be null
	 * 
	 * @return new {@link MethodSignature} with binding set
	 */
	public static Binding makeBinding(
			Map<String, MethodSignature> sigs, MethodCall mc,
			List<Typedef> types, Map<String, TBPResolvedVardef> cvars,
			MethodSignature surroundingMethod) {

		String fullname = mc.getFullname();
		MethodSignature ms = sigs.get(fullname);
		if (ms == null) {
			throw new TBPUndefinedEmitException("Cannot find declaration for: "
					+ fullname, mc);
		}

		Binding result = new Binding(ms);
		
		final int n = ms.getParamNames().size();

		if (n != mc.getParamDecl().size()) {
			throw new TBPResolvingException("Number of parameters for "
					+ mc.getFullname() +
					" does not fit");
		}

		for (int i = 0; i < n; i++) {
			String f_name = ms.getParamNames().get(i);
			String ref_name = mc.getParamDecl().get(i);
			Reference ref = makeReference(ref_name, types, cvars,
					surroundingMethod);

			result.bindParameter(f_name, ref);
		}

		return result;
	}
	
	/**
	 * Creates bindings that refers only to constants.
	 * 
	 * @see #makeBinding(Map, MethodCall, List, Map, MethodSignature)
	 */
	public static Binding makeConstantBinding(Map<String, MethodSignature> sigs, MethodCall mc,
			List<Typedef> types) {
		return makeBinding(sigs, mc, types, null, null);
	}
}
