/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */

package org.ow2.dsrg.fm.tbplib.resolved;

import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbplib.EventTable;
import org.ow2.dsrg.fm.tbplib.TBPResolvingException;
import org.ow2.dsrg.fm.tbplib.ltsa.Automaton;
import org.ow2.dsrg.fm.tbplib.ltsa.AutomatonToDot;
import org.ow2.dsrg.fm.tbplib.ltsa.DeterministicAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.ltsa.NondetermisticAutomaton;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;
import org.ow2.dsrg.fm.tbplib.resolved.util.MethodSignature;
import org.ow2.dsrg.fm.tbplib.resolved.visitor.CheckReturnsVisitor;
import org.ow2.dsrg.fm.tbplib.resolved.visitor.SubstituteVisitor;
import org.ow2.dsrg.fm.tbplib.resolved.visitor.TranslateImperativeVisitor;
import org.ow2.dsrg.fm.tbplib.resolved.visitor.TranslateProvisionVisitor;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

public class ResolvedComponentSpecification {
	private static final boolean DEBUG = false;
	
	private static final TranslateImperativeVisitor translateImperativeVisitor = new TranslateImperativeVisitor();
	private static final TranslateProvisionVisitor translateProvisionVisitor = new TranslateProvisionVisitor();
	
	private String componentname;
    private List<Typedef>  types;
    private Map<String, TBPResolvedVardef> vardefs;
    private List<TBPResolvedProvisionNode> provisions;  
    private List<TBPResolvedMangledReaction> reactions;
    private List<TBPResolvedThreadContainerNode> threads;
    private Map<String, TBPResolvedMethodDefinition> definitions;
    
	public ResolvedComponentSpecification(String componentname,
			List<Typedef> types, Map<String, TBPResolvedVardef> vardefs,
			List<TBPResolvedProvisionNode> provisions,
			List<TBPResolvedThreadContainerNode> threads,
			Map<String, TBPResolvedMethodDefinition> definitions) {
		
		this.componentname = componentname;
		this.types = types;
		this.vardefs = vardefs;
		this.provisions = provisions;
		this.threads = threads;
		this.definitions = definitions;
		this.reactions = mangle_reactions(definitions.values());
		
		CheckReturnsVisitor visitor = new CheckReturnsVisitor();
		for(TBPResolvedMethodDefinition def: definitions.values()){
			// procedure do not contain return
			if(def.getMethodSignature().getReturnType() == null)
				continue;
			TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) def.getChild();
			if(!code.visit(visitor)){
				throw new TBPResolvingException("Not all path contains return in "
						+ def.getMethodSignature().getFullname());
			}
		}
	}	
	
	/**
	 * Make sure that all reaction has no parameters.
	 * @return new list of reactions
	 */
	private List<TBPResolvedMangledReaction> mangle_reactions(Collection<TBPResolvedMethodDefinition> toMangle){
		
		List<TBPResolvedMangledReaction> result_reactions = 
			new ArrayList<TBPResolvedMangledReaction>(toMangle.size());
		
		for (TBPResolvedMethodDefinition decl : toMangle) {
				addAllSubstitution(result_reactions, decl);
		}
		return result_reactions;
	}
	
	// add all full instantiation of definition to result_reactions list
	private void addAllSubstitution(
			List<TBPResolvedMangledReaction> result_reactions,
			TBPResolvedMethodDefinition decl) {
		
		List<List<ConstantRef>> constants = new ArrayList<List<ConstantRef>>();		
		List<Binding> const_bindings = new ArrayList<Binding>();
		
		MethodSignature ms = decl.getMethodSignature();
		final Map<String, Typedef> paramTypes = ms.getParamTypes();
		final List<String> paramNames = ms.getParamNames();
		
		// fill constants 
		for(String name : paramNames){			
			Typedef unbound_type = paramTypes.get(name);
			List<ConstantRef> c = new ArrayList<ConstantRef>(unbound_type.getEnums().size());
			for(String e : unbound_type.getEnums()){
				c.add(new ConstantRef(e,unbound_type)); 
			}
			constants.add(c);	
		}
		
		// take care of definition without parameters 
		// make copy anyway just to be consistent
		if(constants.size() == 0){  
			result_reactions.add(substituteIntoReaction(decl, new Binding(ms)));
			return;
		}
		
		int [] counter = new int[constants.size()];
		int [] limits  = new int[constants.size()];
		
		for(int i = 0; i < limits.length; ++i){
			counter[i] = 0;
			limits[i] = constants.get(i).size();
		}
		
		// Generate all possible bindings
		while(counter[0] < limits[0]){
			
			//sets binding by counter
			Binding binding = new Binding(ms);
			for(int i = 0; i < counter.length; i++){
				binding.bindParameter(paramNames.get(i), constants.get(i).get(counter[i]) );
			}
			assert binding.isBound();
			const_bindings.add(binding);
			
			//add one to counter
			for(int j = counter.length - 1; j >= 0; j--){
				if(++counter[j] < limits[j])
					break;
				else if(j != 0)
					counter[j] = 0;
			}
			
		}
		
		// Generate mangled reactions by substituting binding into decl's code
		for (Binding binding : const_bindings) {
			result_reactions.add(substituteIntoReaction(decl, binding));
		}
			
	}
	
	private TBPResolvedMangledReaction substituteIntoReaction(TBPResolvedMethodDefinition decl,
				Binding binding){
		TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) decl.getChild();
		TBPResolvedImperativeNode code2 = code.visit(new SubstituteVisitor(binding));
		return new TBPResolvedMangledReaction(binding, code2);
	}
	
	public LTSAComponentSpecification makeLTSA(EventTable et) {
		
		Map<String, TBPResolvedVardef> aut_vardefs = 
			new HashMap<String, TBPResolvedVardef>();
		
		for(TBPResolvedVardef v : vardefs.values()){
			aut_vardefs.put(v.getName(), v.clone());
		}
				
		//final Map<String, DeterministicAutomaton> transProvisions = translateProvisions(et);
		final Map<String, NondetermisticAutomaton> transProvisions = translateProvisions(et);

		final Map<Binding, LTSAAutomaton> transMangledReactions = translateMangledReactions();
		final Map<String, LTSAAutomaton> transThreads = translateThreads();
		
		return new LTSAComponentSpecification(componentname,
				types, aut_vardefs, transProvisions,
				transThreads, transMangledReactions);
	}

//	private Map<String, DeterministicAutomaton> translateProvisions(
	private Map<String, NondetermisticAutomaton> translateProvisions(
			EventTable et) {
		// Due to memory issue we don't want to translate NONDET automatons for provisions into 
		// Hack

//		Map<String, DeterministicAutomaton> aut_provisions = 
//		new HashMap<String, DeterministicAutomaton>();

		Map<String, NondetermisticAutomaton> aut_provisions = 
			new HashMap<String, NondetermisticAutomaton>();
				
		for (TBPResolvedProvisionNode n : provisions) {
			final NondetermisticAutomaton nfa = n.visit(translateProvisionVisitor);
			final String provisionName = ((TBPResolvedProvisionContainerNode) n).getName();

			if (DEBUG) {
				System.out.println("Processing " + ((TBPResolvedProvisionContainerNode) n).getName() + " provision");
				System.out.println("  NonDet automaton - states:" + nfa.getStatesCount() + ", edges:" + nfa.getEdgesCount());
				saveAutomaton( provisionName + "NonDet", nfa, et);
			}

			// Due to memory issue we don't want to translate NONDET automatons for provisions into 
			// Hack
			if (true) { 
				aut_provisions.put(	((TBPResolvedProvisionContainerNode) n).getName(), nfa);
				continue;
			}

			DeterministicAutomaton da = nfa.determinize();
			if (DEBUG) {
				System.out.println("  Det automaton - states:" + da.getStatesCount() + ", edges:" + da.getEdgesCount());
				saveAutomaton( provisionName + "Det", da, et);
			}

			DeterministicAutomaton min = da.makeMinimal();
			if (DEBUG) {
				System.out.println("  Min det automaton - states:" + min.getStatesCount() + ", edges:" + min.getEdgesCount());
				saveAutomaton( provisionName + "DetMin", da, et);
			}
			//aut_provisions.put(	((TBPResolvedProvisionContainerNode) n).getName(), min);
		}
		return aut_provisions;
	}
	
	private Map<String, LTSAAutomaton> translateThreads() {
		Map<String, LTSAAutomaton> result = new HashMap<String, LTSAAutomaton>();
		for(TBPResolvedThreadContainerNode n : threads){
			TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) n.getChild();
			final LTSAAutomaton t = code.visit(translateImperativeVisitor);
			result.put(n.getName(), t);
		}
		return result;
	}
	
	// translate definitions - not used - fully substituted reactions are
	// passed to LTSAComponentSpecification instead
	@SuppressWarnings("unused")
	@Deprecated
	private Map<String, LTSAAutomaton> translateReactions() {
		Map<String, LTSAAutomaton> result = new HashMap<String, LTSAAutomaton>();

		for (Entry<String, TBPResolvedMethodDefinition> e : definitions.entrySet()) {
			TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) e.getValue().getChild();
			final LTSAAutomaton t = code.visit(translateImperativeVisitor);
			result.put(e.getKey(), t);
		}
		return result;
	}
	
	private Map<Binding, LTSAAutomaton> translateMangledReactions() {
		Map<Binding, LTSAAutomaton> result = new HashMap<Binding, LTSAAutomaton>();

		for (TBPResolvedMangledReaction r : reactions) {
			final LTSAAutomaton t = r.visit(translateImperativeVisitor); 
			result.put(r.getBinding(), t);
		}
		return result;
	}
	
	
	public String getComponentname() {
		return componentname;
	}
	
	public List<Typedef> getTypes() {
		return types;
	}

	public Map<String, TBPResolvedVardef> getVardefs() {
		return vardefs;
	}
	
	public Map<String, TBPResolvedMethodDefinition> getDefinitions() {
		return definitions;
	}
	
	public List<TBPResolvedProvisionNode> getProvisions() {
		return provisions;
	}

	public List<TBPResolvedMangledReaction> getReactions() {
		return reactions;
	}
	
	public List<TBPResolvedThreadContainerNode> getThreads() {
		return threads;
	}
	
	private static void saveAutomaton(String filename, Automaton aut, EventTable et) {
		if(!filename.endsWith(".dot")){
			filename = filename + ".dot";
		}
		try {
			FileOutputStream stream = new FileOutputStream(filename);
			AutomatonToDot.toDot(aut, et, stream);
			stream.close();
		} catch (IOException e) {
			System.out.println(e);
		}
	} 
}
