package org.ow2.dsrg.fm.tbplib.ltsa.util;

import java.io.PrintStream;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbplib.ltsa.Edge;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.ltsa.State;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedCondition;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedImperativeNull;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;

/**
 * Filters (removes) specific edges from all reaction and thread automatons.
 * Removes specified edges from all automatons but preserve all path in the automatons (other edges are redirected accordingly).
 * 
 * <p>Note: Used to remove NULL, (no edge action - null) and ? (non-determinism choice) labeled edges
 * 
 * @author Alf
 */
public class ComponentReactionEdgeFiter {
	private static final boolean DEBUG = false;
	private final PrintStream out = System.out;
	
	public static interface EdgeTester {
		/**
		 * @param e Edge to check. Cannot be null.
		 * @return True if specified edge should by removed from the automaton.
		 */
		public boolean removeEdge(Edge e);
	};
	
	/**
	 * Removes NULL, (no edge action - null) and ? (non-determinism choice) labeled edges
	 */
	public static class EmptyEdgeRemover implements EdgeTester {
		
		@Override
		public boolean removeEdge(Edge e) {
			assert e != null;
			
			Object eData = e.getData();
			//no edge action - NULL edge
			if (eData == null) {
				return true;
			}
			// Imperative NULL edge
			if (eData instanceof TBPResolvedImperativeNull) {
				return true;
			}
			// '?' edge
			if (eData instanceof TBPResolvedCondition) {
				TBPResolvedCondition cond = (TBPResolvedCondition)eData;
				
				return cond.isNondeterministic();
			}
			return false;
		}
	}
	
	
	private final EdgeTester edgeTester; /// Decide which edge to remove and which preserve.
	
	/**
	 * Creates filter which removes empty edges.
	 */
	public ComponentReactionEdgeFiter() {
		this( new EmptyEdgeRemover());
	}
	
	/**
	 * Create filter which uses user spefied edge tester/
	 * @param eTester Edge tester to use.
	 */
	public ComponentReactionEdgeFiter(EdgeTester eTester) {
		this.edgeTester = eTester;
	}

	/**
	 * Modifies all (reaction a thread) automatons is given specification.
	 *  Removes edges according {@link EdgeTester}
	 *  
	 * @param comp Specification to process. Cannot be null.
	 */
	public void processSpecification(LTSAComponentSpecification comp) {
		assert comp != null;
		
		for(Entry<Binding, LTSAAutomaton> automatonPair : comp.getReactions().entrySet()) {
			if (DEBUG) { out.println(this.getClass().getSimpleName() + ".processSpecification() - Processing reaction " + automatonPair.getKey()); }
			processAutomaton(automatonPair.getValue());
		}
		
		for(Entry<Binding, LTSAAutomaton> automatonPair : comp.getReactions().entrySet()) {
			if (DEBUG) { out.println(this.getClass().getSimpleName() + ".processSpecification() - Processing thread " + automatonPair.getKey()); }
			processAutomaton(automatonPair.getValue());
		}
	}

	private void processAutomaton(LTSAAutomaton automaton) {
		Set<State> processedStates = new HashSet<State>();
		
		processAutomatonState(automaton.getStart(), processedStates);
	}
	private void processAutomatonState(State state, Set<State> processedStates) {
		/*
		 * What we do: ? is edge to remove, * is any edge
		 *   A -?-> B -*-> C
		 *           \-*-> D
		 *  is converted into (remove -*- edge and all path throws this edge are appended into A)
		 *  A -*-> C
		 *   \-*-> D
		 */
		
		// Try to remove all outgoing null Edges
		List<Edge> stateEdges = state.getEdges();
		for(int i = 0; i < stateEdges.size(); i++) {
			Edge e = stateEdges.get(i); // Cannot use iterator because new edges are appended in the end of the list
			if (edgeTester.removeEdge(e)) {
				// Edge to remove ... try all all successors of this the target state (to preserve all paths)
				State targetState = e.getTarget();
				if (targetState.equals(state)) {
					// Self cycle skip this edge
					continue;
				}
				for(Edge e1 : targetState.getEdges()) {
					if(edgeTester.removeEdge(e1) && e1.getTarget().equals(e1.getSource())) {
						continue;
						// Skip this simple cycle
					}
					// [State]-e->[targetState]-e1->[x?] 
					// add new (transitive) edge [state]-(e1Copy)->[x?]
					state.addEdge(e1.getData(), e1.getTarget());
				}
			} // edge to remove
		} // for all outgoing edges
		
		// Filter out all edges to remove
		int processedEdges = 0;
		while(processedEdges < stateEdges.size()) {
			Edge e2 = stateEdges.get(processedEdges);
			if (edgeTester.removeEdge(e2)) {
				stateEdges.remove(processedEdges);
				// Don't increment index, we wanted to check new edge on this position
			} else {
				processedEdges++;
			}
		}
		
		processedStates.add(state);
		//Recursively process all outgoing edges
		for(Edge e3 : stateEdges) {
			// if target not processed until now -> recursively call them (DFS traversal)
			if (!processedStates.contains(e3.getTarget())) {
				processAutomatonState(e3.getTarget(), processedStates);
			}
		}
	}
	
}
