/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.ltsa;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.dsrg.fm.tbplib.resolved.CVRef;
import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.LastCallRef;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedAssignment;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedCondition;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedReturn;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedValue;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedEmit;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedTau;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;
import org.ow2.dsrg.fm.tbplib.resolved.util.MethodSignature;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Contains static methods that work on LTSAAutomatons.
 * 
 * @author caitt3am
 */
public class Utils {

	public static List<Edge> findEmits(State s){
		final List<Edge> result = new ArrayList<Edge>();
		Utils.processGraph(s, new EdgeVisitor(){
			
			@Override
			public void visitEdge(Edge e) {
				if (e.getData() instanceof TBPResolvedEmit) {
					result.add(e);
				}
			}
			
		});
		return result;
	}
	
	public static List<Edge> findReturns(State s){
		final List<Edge> result = new ArrayList<Edge>();
		Utils.processGraph(s, new EdgeVisitor(){
			
			@Override
			public void visitEdge(Edge e) {
				if (e.getData() instanceof TBPResolvedReturn) {
					result.add(e);
				}
			}
			
		});
		return result;
	} 

	/**
	 * Visits all edges of graph reachable from s.
	 * @param s - Where to start
	 * @param v - this object is passed every visited edge 
	 */
	public static void processGraph(State s, EdgeVisitor v){
		
		List<State> node_list = new ArrayList<State>();
		s.Mark();
		node_list.add(s);
		
		for(int index = 0; index < node_list.size(); index++){
			s = node_list.get(index);
			for(Edge e : s.getEdges()){
				assert e.getSource() == s : "Inconsistent edge";
				v.visitEdge(e);
				if(!e.getTarget().isMarked()){
					e.getTarget().Mark();
					node_list.add(e.getTarget());
				}
			}
		}
		
		for (State state : node_list) {
			state.Unmark();
		}
	}
	
	public static State copyGraph(State s){
		final Map<State,State> map = new HashMap<State, State>();
		processGraph(s, new EdgeVisitor(){

			@Override
			public void visitEdge(Edge e) {
				
				State old_target = e.getTarget();
				State old_source = e.getSource();
				
				State new_target = map.get(old_target);
				State new_source = map.get(old_source);
				
				if(new_target == null){
					new_target = new State();
					map.put(old_target, new_target);
				}
				
				if(new_source == null){
					new_source = new State();
					map.put(old_source, new_source);
				}
				
				new_source.addEdge(e.getData(), new_target);
			}
			
		});
		
		return map.get(s);
	} 

	/**
	 * This functions inplace inlines given emit.
	 * 
	 * @param e - Edge containing emit to be inlined
	 * @param defs - map from reactions bindings to theirs automatons
	 * @param inlined_defs - map from reactions bindings to theirs automatons, 
	 * that are known to be fully inlined
	 */
	public static void inlineEmit(Edge e, 
			Map<Binding, LTSAAutomaton> defs,
			Map<Binding, LTSAAutomaton> inlined_defs){
		inlineEmit(e, defs, inlined_defs, new ArrayDeque<Binding>());
	}
	
	private static void inlineEmit(Edge e, 
			Map<Binding, LTSAAutomaton> defs,
			Map<Binding, LTSAAutomaton> inlined_defs,
			Deque<Binding> stack){

		State target = e.getTarget();
		List<Edge> edges = target.getEdges();
		
		if(edges.size() == 0){
			inlineBareEmit(e, defs, inlined_defs, stack);
			return;
		}
		
		Edge edge = edges.get(0);
		Object data = edge.getData();
		if(data instanceof TBPResolvedCondition){
			TBPResolvedCondition cond = (TBPResolvedCondition) data;
			if(cond.getLeft() instanceof LastCallRef){
				inlineEmitIntoSwitch(e, edges, defs, inlined_defs, stack);
				return;
			} else {
				checkNotLastCallCondition(edges);
				inlineBareEmit(e, defs, inlined_defs, stack);
				return;
			}
		}
		
		if(data instanceof TBPResolvedAssignment){
			TBPResolvedAssignment ass = (TBPResolvedAssignment) data;
			
			assert ass.getValue().isReference();
			assert edges.size() == 1;
			
			if(ass.getValue().getReference() instanceof LastCallRef){
				inlineEmitIntoAssign(e, edge, defs, inlined_defs, stack);
				return;
			}
		}	
		
		inlineBareEmit(e, defs, inlined_defs, stack);
				
	}

	private static void checkNotLastCallCondition(List<Edge> list) {
		for(Edge e : list){
			if (e.getData() instanceof TBPResolvedCondition) {
				TBPResolvedCondition cond = (TBPResolvedCondition) e.getData();
				if(cond.getLeft() instanceof LastCallRef){
					throw new RuntimeException("Condition referring to result of last call not expected");
				}
			}
		}
	}
	
	/**
	 * Translate edge with condition to mapping from constant in that condition to target of given edge.  
	 */
	private static Map<ConstantRef, State> convertConditions(List<Edge> edges) {
		Map<ConstantRef, State> result = new HashMap<ConstantRef, State>();
		for(Edge e : edges){
			if(!(e.getData() instanceof TBPResolvedCondition))
				throw new RuntimeException("Inlining emit into switch and non-guarded edge detected");
			TBPResolvedCondition cond = (TBPResolvedCondition) e.getData();
			if(!(cond.getLeft() instanceof LastCallRef))
				throw new RuntimeException("Inlining emit into switch and only some guards use last call");
			result.put(cond.getRight(), e.getTarget());
		}
		return result;
	}

	/**
	 * Construct condition for given substitution.
	 * @param names Names parameters that were substituted
	 * @param const_binding Binding where all non-const references were replaced
	 * @param orig_binding Original binding
	 * @return condition that is satisfied if call matches const_binging 
	 */
	private static TBPResolvedCondition getCase(List<String> names,	
			Binding const_binding, Binding orig_binding) {
		List<ConstantRef> cs = new ArrayList<ConstantRef>();
		for(String name : names){
			cs.add((ConstantRef) const_binding.getValue(name));
		}
		List<CVRef> cvs = new ArrayList<CVRef>();
		for(String name : names){
			cvs.add((CVRef) orig_binding.getValue(name));
		}
		return new TBPResolvedCondition(cvs, cs);
	}	

	private static void inlineEmitIntoSwitch(Edge e, List<Edge> edges,
			Map<Binding, LTSAAutomaton> defs,
			Map<Binding, LTSAAutomaton> inlined_defs,
			Deque<Binding> stack) {
		final Map<ConstantRef, State> branches = convertConditions(edges);
		inline(e, defs, inlined_defs, stack, new SwitchEmitProcessor(branches));		
	}

	private static void inlineEmitIntoAssign(Edge e, Edge edge,
			Map<Binding, LTSAAutomaton> defs,
			Map<Binding, LTSAAutomaton> inlined_defs,
			Deque<Binding> stack) {
		
		Edge sec = e.getTarget().getEdges().get(0); 
		inline(e, defs, inlined_defs, stack, 
				new AssignEmitProcessor(sec.getTarget(), (TBPResolvedAssignment) sec.getData()));
	}

	private static void inlineBareEmit(Edge e,
			Map<Binding, LTSAAutomaton> defs,
			Map<Binding, LTSAAutomaton> inlined_defs,
			Deque<Binding> stack) {
		inline(e, defs, inlined_defs, stack, new BareEmitProcessor(e.getTarget()));
	}


	private interface ReturnProcessor {
		void processReturn(Edge e, Binding const_binding);
	}

	private static class BareEmitProcessor implements ReturnProcessor {
		private State target;
		
		public BareEmitProcessor(State target) {
			this.target = target;
		}

		@Override
		public void processReturn(Edge e, Binding const_binding) {
			TBPResolvedReturn ret = (TBPResolvedReturn) e.getData();
			e.setData(new TBPResolvedTau(const_binding, ret.getReturnValue()));
			e.setTarget(target);
		}
		
	}
	
	private static class SwitchEmitProcessor implements ReturnProcessor {
		private final Map<ConstantRef, State> branches;
		
		public SwitchEmitProcessor(Map<ConstantRef, State> branches) {
			this.branches = branches;
		}

		@Override
		public void processReturn(Edge e, Binding const_binding) {
			TBPResolvedReturn ret = (TBPResolvedReturn) e.getData();
			e.setData(new TBPResolvedTau(const_binding, ret.getReturnValue()));

			final State target = branches.get(ret.getReturnValue());
			e.setTarget(target);
		}	
	}
	
	private static class AssignEmitProcessor implements ReturnProcessor {
		private State target;
		private TBPResolvedAssignment ass;
		
		public AssignEmitProcessor(State target, TBPResolvedAssignment ass) {
			this.target = target;
			this.ass = ass;
		}

		@Override
		public void processReturn(Edge e, Binding const_binding) {

			TBPResolvedReturn ret = (TBPResolvedReturn) e.getData();
			State ns = new State();
			e.setData(new TBPResolvedTau(const_binding, ret.getReturnValue()));
			e.setTarget(ns);
			ns.addEdge(
					new TBPResolvedAssignment(ass.getIdf(), new TBPResolvedValue(ret.getReturnValue())), 
					target);
		}
	
		
	}
	
	/**
	 * Return inlined automaton for given binding. 
	 *
	 * @param const_binding - for what binding we want automaton
	 * @param defs - non-inlined automatons
	 * @param inlined_defs - inlined automatons
	 * @param stack - keep track of currently visited automatons to avoid cycling
	 * @return inlined automaton for given binding.
	 */
	private static LTSAAutomaton getInlinedAutomaton(Binding const_binding,
			Map<Binding, LTSAAutomaton> defs,
			Map<Binding, LTSAAutomaton> inlined_defs,
			Deque<Binding> stack){
		if(stack.contains(const_binding)){
			throw new RuntimeException(const_binding + " contains recursive call to itself. Cannot inline. ");
		}
		LTSAAutomaton result = inlined_defs.get(const_binding);
		if(result != null)
			return result;
		
		stack.add(const_binding);
		result = defs.get(const_binding);
		
		List<Edge> emits = Utils.findEmits(result.getStart());
		for(Edge e: emits){
			Utils.inlineEmit(e, defs, inlined_defs, stack);
		}
		inlined_defs.put(const_binding, result);
		
		Binding remove = stack.removeLast();
		assert remove == const_binding;
		return result;
	}
	private static void inline(Edge e,
			Map<Binding, LTSAAutomaton> defs,
			Map<Binding, LTSAAutomaton> inlined_defs,
			Deque<Binding> stack,
			ReturnProcessor proc) {
		
		TBPResolvedEmit emit = (TBPResolvedEmit) e.getData();
		Binding orig_binding = emit.getBinding();
				
		final List<String> paramNames = orig_binding.getMethodSignature().getParamNames();
		
		List<String> nonconst_names = orig_binding.getParameterNamesNotBoundToConstant();
		List<String> const_names = new ArrayList<String>(paramNames);
		const_names.removeAll(nonconst_names);
				
		State source = e.getSource();
		source.removeEdge(e);
		
		List<Binding> bindings = getAllConstantBindings(orig_binding);
		for(final Binding const_binding : bindings){
			
			LTSAAutomaton automaton = getInlinedAutomaton(const_binding, defs, inlined_defs, stack); 
			State copyGraph = copyGraph(automaton.getStart());
			final List<Edge> ret_edges = findReturns(copyGraph); 
			
			for(Edge e2 : ret_edges){
				proc.processReturn(e2, const_binding);
			}
			
			if(bindings.size() > 1){
				TBPResolvedCondition case1 = getCase(nonconst_names, const_binding, orig_binding);	
				source.addEdge(case1, copyGraph);
			} else {
				for(Edge e3 : copyGraph.getEdges()){
					source.addEdge(e3.getData(), e3.getTarget());
				}
			}
		}
		
	}
	
	private static List<Binding> getAllConstantBindings(Binding b) {
		
		List<List<ConstantRef>> constants = new ArrayList<List<ConstantRef>>();		
		List<Binding> const_bindings = new ArrayList<Binding>();
		
	
		MethodSignature ms = b.getMethodSignature();
		final Map<String, Typedef> paramTypes = ms.getParamTypes();
		final List<String> paramNames = b.getParameterNamesNotBoundToConstant();
		
		// fill constants 
		for(String name : paramNames){			
			Typedef unbound_type = paramTypes.get(name);
			List<ConstantRef> c = new ArrayList<ConstantRef>(unbound_type.getEnums().size());
			for(String e : unbound_type.getEnums()){
				c.add(new ConstantRef(e,unbound_type)); 
			}
			constants.add(c);	
		}
		
		// take care of case when all paremeters are constant
		if(constants.size() == 0){  
			const_bindings.add(b);
			return const_bindings;
		}
		
		int [] counter = new int[constants.size()];
		int [] limits  = new int[constants.size()];
		
		for(int i = 0; i < limits.length; ++i){
			counter[i] = 0;
			limits[i] = constants.get(i).size();
		}
		
		// Generate all possible bindings
		while(counter[0] < limits[0]){
			
			//sets binding by counter
			Binding binding = b.extractConstantBinding();
			for(int i = 0; i < counter.length; i++){
				binding.bindParameter(paramNames.get(i), constants.get(i).get(counter[i]) );
			}
			assert binding.isBound();
			const_bindings.add(binding);
			
			//add one to counter
			for(int j = counter.length - 1; j >= 0; j--){
				if(++counter[j] < limits[j])
					break;
				else if(j != 0)
					counter[j] = 0;
			}
			
		}
		
		return const_bindings;
	}
}
