/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.ltsa;

import java.util.List;

import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedCondition;

/**
 * Automaton representing imperative code.
 * 
 * Its edges are labeled with events, guards or test_and_set.
 * 
 * For now, events are represented by TBPResolved (unary) nodes,
 * guards by {@link TBPResolvedCondition} and test_and_set by
 * {@link ConstantRef} to mutex variable. 
 * 
 * Edge can have null label (to make some operation easier 
 * to implement). 
 * 
 * @author caitt3am
 *
 */
public class LTSAAutomaton {
	
	private State start;
	private State fin; // helper state to facilitate appending  

	private LTSAAutomaton(){
		start = new State();
		fin = start;
	}
	
	private LTSAAutomaton(Object edgeData){
		start = new State();
		fin = new State();
		start.addEdge(edgeData, fin);
	}
	
	public State getStart() {
		return start;
	}

	/**
	 * Append given automaton to this one (this is modified). 
	 * @param automat - what to append
	 * @return this
	 */
	public LTSAAutomaton append(LTSAAutomaton automat) {
		for(Edge e : automat.start.getEdges())
			fin.addEdge(e.getData(), e.getTarget());
		fin = automat.fin;
		return this;
	}
	
	public static LTSAAutomaton createSimple(Object edgeData) {
		return new LTSAAutomaton(edgeData);
	}

	/**
	 * Make automaton for if statement
	 * @param condition 
	 * @param automaton - if branch
	 * @param automaton2 - else branch, can be <code>null</code>
	 * @return automaton for if statement
	 */
	public static LTSAAutomaton createChoice(TBPResolvedCondition condition,
			LTSAAutomaton automaton, LTSAAutomaton automaton2) {

		LTSAAutomaton result = new LTSAAutomaton(condition);
		result.append(automaton);
		
		if(automaton2 == null){
			result.start.addEdge(condition.createNegation(), result.fin);
		} else {
			result.start.addEdge(condition.createNegation(), automaton2.start);
			State new_fin = new State();
			automaton2.fin.addNullEdge(new_fin);
			result.fin.addNullEdge(new_fin);
			result.fin = new_fin;
		}
		return result;
	}

	/**
	 * Make automaton that chooses from multiple automatons.  
	 * @param conditions - guards for branches
	 * @param branches - list of automatons
	 */
	public static LTSAAutomaton createMultipleChoice(List<TBPResolvedCondition> conditions,
			List<LTSAAutomaton> branches) {
		
		LTSAAutomaton result = new LTSAAutomaton();
		result.fin = new State();
		
		final int s = branches.size();
		for(int i = 0; i < s; i++){
			LTSAAutomaton branch = branches.get(i);
			TBPResolvedCondition cond = conditions.get(i);
			
			result.start.addEdge(cond, branch.start);
			branch.fin.addNullEdge(result.fin);
		}
		
		return result;
	}
	
	/**
	 * Make automaton that cycles while condition is satisfied.
	 * @param condition - guard for body 
	 * @param a - body of while cycle
	 */
	public static LTSAAutomaton createWhile(TBPResolvedCondition condition,	
			LTSAAutomaton a) {
		
		LTSAAutomaton result = new LTSAAutomaton(condition.createNegation());
		result.start.addEdge(condition, a.start);
		a.fin.addNullEdge(result.start);
		
		return result;
	}

	/**
	 * @return Gets final state of automaton
	 */
	public State getFin() {
		return fin;
	}

}
