/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.jpf.tools;

import gov.nasa.jpf.Config;
import gov.nasa.jpf.JPF;
import gov.nasa.jpf.ListenerAdapter;
import gov.nasa.jpf.jvm.AnnotationInfo;
import gov.nasa.jpf.jvm.ClassInfo;
import gov.nasa.jpf.jvm.ExceptionHandler;
import gov.nasa.jpf.jvm.JVM;
import gov.nasa.jpf.jvm.MethodInfo;
import gov.nasa.jpf.jvm.ThreadInfo;
import gov.nasa.jpf.jvm.bytecode.GOTO;
import gov.nasa.jpf.jvm.bytecode.IfInstruction;
import gov.nasa.jpf.jvm.bytecode.Instruction;
import gov.nasa.jpf.jvm.bytecode.InvokeInstruction;
import gov.nasa.jpf.jvm.bytecode.ReturnInstruction;
import gov.nasa.jpf.report.ConsolePublisher;
import gov.nasa.jpf.report.Publisher;
import gov.nasa.jpf.report.PublisherExtension;
import gov.nasa.jpf.util.Misc;
import gov.nasa.jpf.util.StringSetMatcher;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.logging.Logger;

public class CoverageAnalyzer
extends ListenerAdapter
implements PublisherExtension {
    static Logger log = JPF.getLogger("gov.nasa.jpf.tools.CoverageAnalyzer");
    StringSetMatcher includes = null;
    StringSetMatcher excludes = null;
    StringSetMatcher loaded;
    static boolean loadedOnly;
    static boolean showMethods;
    static boolean showMethodBodies;
    static boolean excludeHandlers;
    static boolean showBranchCoverage;
    static boolean showRequirements;
    HashMap<String, ClassCoverage> classes = new HashMap();
    String lastStart;
    MethodInfo lastMi = null;
    MethodCoverage lastMc = null;
    HashMap<String, HashSet<MethodCoverage>> requirements;

    public CoverageAnalyzer(Config conf, JPF jpf) {
        this.includes = StringSetMatcher.getNonEmpty(conf.getStringArray("coverage.include"));
        this.excludes = StringSetMatcher.getNonEmpty(conf.getStringArray("coverage.exclude"));
        showMethods = conf.getBoolean("coverage.show_methods", false);
        showMethodBodies = conf.getBoolean("coverage.show_bodies", false);
        excludeHandlers = conf.getBoolean("coverage.exclude_handlers", false);
        showBranchCoverage = conf.getBoolean("coverage.show_branches", true);
        loadedOnly = conf.getBoolean("coverage.loaded_only", false);
        showRequirements = conf.getBoolean("coverage.show_requirements", false);
        if (!loadedOnly) {
            this.getCoverageCandidates();
        }
        jpf.addPublisherExtension(ConsolePublisher.class, this);
    }

    void getCoverageCandidates() {
        for (String s : ClassInfo.getClassPathElements()) {
            log.fine("analyzing classpath element: " + s);
            File f = new File(s);
            if (!f.exists()) continue;
            if (f.isDirectory()) {
                this.traverseDir(f, null);
                continue;
            }
            if (!s.endsWith(".jar")) continue;
            this.traverseJar(f);
        }
    }

    void addClassEntry(String clsName) {
        ClassCoverage cc = new ClassCoverage(clsName);
        this.classes.put(clsName, cc);
        log.info("added class candidate: " + clsName);
    }

    boolean isAnalyzedClass(String clsName) {
        return StringSetMatcher.isMatch(clsName, this.includes, this.excludes);
    }

    void traverseDir(File dir, String pkgPrefix) {
        for (File f : dir.listFiles()) {
            if (f.isDirectory()) {
                String prefix = f.getName();
                if (pkgPrefix != null) {
                    prefix = pkgPrefix + '.' + prefix;
                }
                this.traverseDir(f, prefix);
                continue;
            }
            String fname = f.getName();
            if (!fname.endsWith(".class")) continue;
            if (f.canRead() && f.length() > 0L) {
                String clsName = fname.substring(0, fname.length() - 6);
                if (pkgPrefix != null) {
                    clsName = pkgPrefix + '.' + clsName;
                }
                if (!this.isAnalyzedClass(clsName)) continue;
                this.addClassEntry(clsName);
                continue;
            }
            log.warning("cannot read class file: " + fname);
        }
    }

    void traverseJar(File jar) {
        try {
            JarFile jf = new JarFile(jar);
            Enumeration<JarEntry> entries = jf.entries();
            while (entries.hasMoreElements()) {
                String eName;
                JarEntry e = entries.nextElement();
                if (e.isDirectory() || !(eName = e.getName()).endsWith(".class")) continue;
                if (e.getSize() > 0L) {
                    String clsName = eName.substring(0, eName.length() - 6);
                    if (!this.isAnalyzedClass(clsName = clsName.replace('/', '.'))) continue;
                    this.addClassEntry(clsName);
                    continue;
                }
                log.warning("cannot read jar entry: " + eName);
            }
        }
        catch (IOException iox) {
            iox.printStackTrace();
        }
    }

    void printCoverageStatistics(PrintWriter pw) {
        ArrayList<Map.Entry<String, ClassCoverage>> clsEntries = Misc.createSortedEntryList(this.classes, new Comparator<Map.Entry<String, ClassCoverage>>(){

            @Override
            public int compare(Map.Entry<String, ClassCoverage> o1, Map.Entry<String, ClassCoverage> o2) {
                return o2.getKey().compareTo(o1.getKey());
            }
        });
        this.printClassCoverages(pw, clsEntries);
        if (showRequirements) {
            this.printRequirementsCoverage(pw);
        }
    }

    void printRequirementsCoverage(PrintWriter pw) {
        HashMap<String, Integer> reqMethods = this.getGlobalRequirementsMethods();
        String space = "  ";
        Coverage bbAll = new Coverage(0, 0);
        Coverage insnAll = new Coverage(0, 0);
        Coverage branchAll = new Coverage(0, 0);
        Coverage mthAll = new Coverage(0, 0);
        Coverage reqAll = new Coverage(0, 0);
        pw.println();
        pw.println();
        reqAll.total = reqMethods.size();
        mthAll.total = this.computeTotalRequirementsMethods(reqMethods);
        pw.println("--------------------------------- requirements coverage -----------------------------------");
        pw.println("bytecode            basic-block         branch              methods             requirement");
        pw.println("-------------------------------------------------------------------------------------------");
        for (String id : Misc.getSortedKeyStrings(reqMethods)) {
            Coverage bbCoverage = new Coverage(0, 0);
            Coverage insnCoverage = new Coverage(0, 0);
            Coverage branchCoverage = new Coverage(0, 0);
            Coverage reqMth = new Coverage(reqMethods.get(id), 0);
            if (this.requirements != null && this.requirements.containsKey(id)) {
                ++reqAll.covered;
                for (MethodCoverage mc : this.requirements.get(id)) {
                    insnCoverage.add(mc.getCoveredInsn());
                    bbCoverage.add(mc.getCoveredBasicBlocks());
                    branchCoverage.add(mc.getCoveredBranches());
                    ++mthAll.covered;
                    ++reqMth.covered;
                }
                pw.print(insnCoverage.format());
                pw.print(space);
                pw.print(bbCoverage.format());
                pw.print(space);
                pw.print(branchCoverage.format());
                pw.print(space);
                pw.print(reqMth.format());
                pw.print("\"" + id + "\"");
                pw.println();
                if (showMethods) {
                    for (MethodCoverage mc : this.requirements.get(id)) {
                        pw.print("  ");
                        pw.print(mc.getCoveredInsn().format());
                        pw.print(space);
                        pw.print(mc.getCoveredBasicBlocks().format());
                        pw.print(space);
                        pw.print(mc.getCoveredBranches().format());
                        pw.print(space);
                        pw.println(mc.getMethodInfo().getFullName());
                    }
                }
            } else {
                pw.print(" -                   -                   -                  ");
                pw.print(reqMth.format());
                pw.print("\"" + id + "\"");
                pw.println();
            }
            insnAll.add(insnCoverage);
            bbAll.add(bbCoverage);
            branchAll.add(branchCoverage);
        }
        pw.println();
        pw.println("------------------------------------------------------------------------------------------");
        pw.print(insnAll.format());
        pw.print(space);
        pw.print(bbAll.format());
        pw.print(space);
        pw.print(branchAll.format());
        pw.print(space);
        pw.print(mthAll.format());
        pw.print(space);
        pw.print(reqAll.format());
        pw.println(" total");
    }

    HashMap<String, Integer> getGlobalRequirementsMethods() {
        HashMap<String, Integer> map = new HashMap<String, Integer>();
        for (ClassCoverage cc : this.classes.values()) {
            ClassInfo ci = ClassInfo.getClassInfo(cc.className);
            for (MethodInfo mi : ci.getDeclaredMethodInfos()) {
                AnnotationInfo ai = this.getRequirementsAnnotation(mi);
                if (ai == null) continue;
                for (String id : ai.getValueAsStringArray()) {
                    Integer n = map.get(id);
                    if (n == null) {
                        map.put(id, 1);
                        continue;
                    }
                    map.put(id, n + 1);
                }
            }
        }
        return map;
    }

    int computeTotalRequirementsMethods(HashMap<String, Integer> map) {
        int n = 0;
        for (Integer i : map.values()) {
            n += i.intValue();
        }
        return n;
    }

    void printClassCoverages(PrintWriter pw, List<Map.Entry<String, ClassCoverage>> clsEntries) {
        String space = "  ";
        Coverage clsCoverage = new Coverage(0, 0);
        Coverage mthCoverage = new Coverage(0, 0);
        Coverage bbCoverage = new Coverage(0, 0);
        Coverage insnCoverage = new Coverage(0, 0);
        Coverage branchCoverage = new Coverage(0, 0);
        pw.println();
        pw.println("----------------------------------- class coverage ---------------------------------------");
        pw.println("bytecode            basic-block         branch              methods             class");
        pw.println("------------------------------------------------------------------------------------------");
        for (Map.Entry<String, ClassCoverage> e : clsEntries) {
            ClassCoverage cc = e.getValue();
            if (cc.covered) {
                ++clsCoverage.covered;
            } else if (cc.isInterface()) continue;
            ++clsCoverage.total;
            Coverage cov = cc.getCoveredInsn();
            insnCoverage.add(cov);
            pw.print(cov.format());
            pw.print(space);
            cov = cc.getCoveredBasicBlocks();
            bbCoverage.add(cov);
            pw.print(cov.format());
            pw.print(space);
            cov = cc.getCoveredBranches();
            branchCoverage.add(cov);
            pw.print(cov.format());
            pw.print(space);
            cov = cc.getCoveredMethods();
            mthCoverage.add(cov);
            pw.print(cov.format());
            pw.print(space);
            pw.println(e.getKey());
            if (!showMethods) continue;
            this.printMethodCoverages(pw, cc);
            pw.println();
        }
        pw.println("------------------------------------------------------------------------------------------");
        pw.print(insnCoverage.format());
        pw.print(space);
        pw.print(bbCoverage.format());
        pw.print(space);
        pw.print(branchCoverage.format());
        pw.print(space);
        pw.print(mthCoverage.format());
        pw.print(space);
        pw.print(clsCoverage.format());
        pw.println(" total");
    }

    void printMethodCoverages(PrintWriter pw, ClassCoverage cc) {
        if (cc.methods == null) {
            return;
        }
        ArrayList<Map.Entry<MethodInfo, MethodCoverage>> mthEntries = Misc.createSortedEntryList(cc.methods, new Comparator<Map.Entry<MethodInfo, MethodCoverage>>(){

            @Override
            public int compare(Map.Entry<MethodInfo, MethodCoverage> o1, Map.Entry<MethodInfo, MethodCoverage> o2) {
                int b;
                int a = o2.getValue().getCoveredInsn().percent();
                if (a == (b = o1.getValue().getCoveredInsn().percent())) {
                    return o2.getKey().getUniqueName().compareTo(o1.getKey().getUniqueName());
                }
                return a - b;
            }
        });
        for (Map.Entry<MethodInfo, MethodCoverage> e : mthEntries) {
            MethodCoverage mc = e.getValue();
            MethodInfo mi = mc.getMethodInfo();
            Coverage insnCoverage = mc.getCoveredInsn();
            Coverage branchCoverage = mc.getCoveredBranches();
            pw.print("  ");
            pw.print(insnCoverage.format());
            pw.print("  ");
            pw.print(mc.getCoveredBasicBlocks().format());
            pw.print("  ");
            pw.print(branchCoverage.format());
            pw.print("  ");
            pw.println(mi.getLongName());
            if (!showMethodBodies || insnCoverage.isFullyCovered() && branchCoverage.isFullyCovered()) continue;
            this.printBodyCoverage(pw, mc);
        }
    }

    void printBodyCoverage(PrintWriter pw, MethodCoverage mc) {
        int i;
        MethodInfo mi = mc.getMethodInfo();
        Instruction[] code = mi.getInstructions();
        BitSet cov = mc.getExecutedInsn();
        int start = -1;
        BitSet handlers = mc.getHandlers();
        if (excludeHandlers) {
            cov.andNot(handlers);
        }
        for (i = 0; i < code.length; ++i) {
            if (!cov.get(i)) {
                if (start != -1) continue;
                start = i;
                continue;
            }
            if (start == -1) continue;
            this.printSourceRange(pw, code, cov, handlers, start, i - 1, null);
            start = -1;
        }
        if (start != -1) {
            this.printSourceRange(pw, code, cov, handlers, start, i - 1, null);
        }
        BitSet branches = mc.getBranches();
        this.lastStart = null;
        for (i = 0; i < code.length; ++i) {
            if (!branches.get(i)) continue;
            String prefix = "";
            BitSet bTrue = mc.branchTrue;
            BitSet bFalse = mc.branchFalse;
            if (bTrue != null) {
                boolean cTrue = bTrue.get(i);
                boolean cFalse = bFalse.get(i);
                prefix = cTrue ? (cFalse ? null : "F ") : (cFalse ? "T " : "N ");
            } else {
                prefix = "N ";
            }
            if (prefix == null) continue;
            this.printSourceRange(pw, code, cov, handlers, i, i, prefix);
        }
    }

    void printSourceRange(PrintWriter pw, Instruction[] code, BitSet cov, BitSet handlers, int start, int end, String prefix) {
        String locStart = code[start].getFileLocation();
        String locEnd = code[end].getFileLocation();
        if (!locStart.equals(this.lastStart)) {
            this.lastStart = locStart;
            this.printBlanks(pw, 64);
            if (prefix != null) {
                pw.print(prefix);
            }
            pw.print("(");
            pw.print(locStart);
            pw.println(handlers.get(start) ? ")x" : ")");
            if (!locStart.equals(locEnd)) {
                this.printBlanks(pw, 64);
                pw.print("..(");
                pw.print(locEnd);
                pw.println(handlers.get(end) ? ")x" : ")");
            }
        }
    }

    void printBlanks(PrintWriter pw, int n) {
        for (int i = 0; i < n; ++i) {
            pw.print(' ');
        }
    }

    @Override
    public void classLoaded(JVM vm) {
        ClassCoverage cc;
        ClassInfo ci = vm.getLastClassInfo();
        String clsName = ci.getName();
        if (loadedOnly && this.isAnalyzedClass(clsName)) {
            this.addClassEntry(clsName);
        }
        if ((cc = this.classes.get(clsName)) != null) {
            cc.setLoaded(ci);
        }
    }

    MethodCoverage getMethodCoverage(JVM vm) {
        Instruction insn = vm.getLastInstruction();
        if (!insn.isExtendedInstruction()) {
            MethodInfo mi = insn.getMethodInfo();
            if (mi != this.lastMi) {
                ClassCoverage cc;
                this.lastMc = null;
                this.lastMi = mi;
                ClassInfo ci = mi.getClassInfo();
                if (ci != null && (cc = this.classes.get(ci.getName())) != null) {
                    this.lastMc = cc.getMethodCoverage(mi);
                }
            }
            return this.lastMc;
        }
        return null;
    }

    void updateRequirementsCoverage(String[] ids, MethodCoverage mc) {
        if (this.requirements == null) {
            this.requirements = new HashMap();
        }
        for (String id : ids) {
            HashSet<MethodCoverage> mcs = this.requirements.get(id);
            if (mcs == null) {
                mcs = new HashSet();
                this.requirements.put(id, mcs);
            }
            if (mcs.contains(mc)) continue;
            mcs.add(mc);
        }
    }

    AnnotationInfo getRequirementsAnnotation(MethodInfo mi) {
        return mi.getAnnotation("gov.nasa.jpf.Requirement");
    }

    @Override
    public void instructionExecuted(JVM vm) {
        Instruction insn = vm.getLastInstruction();
        MethodCoverage mc = this.getMethodCoverage(vm);
        if (mc != null) {
            AnnotationInfo ai;
            mc.setExecuted(vm.getLastThreadInfo(), insn);
            if (showRequirements && insn.getPosition() == 0 && (ai = this.getRequirementsAnnotation(mc.getMethodInfo())) != null) {
                String[] ids = ai.getValueAsStringArray();
                this.updateRequirementsCoverage(ids, mc);
            }
        }
    }

    @Override
    public void choiceGeneratorSet(JVM vm) {
    }

    @Override
    public void publishFinished(Publisher publisher) {
        PrintWriter pw = publisher.getOut();
        publisher.publishTopicStart("coverage statistics");
        this.printCoverageStatistics(pw);
    }

    static class ClassCoverage {
        String className;
        ClassInfo ci;
        boolean covered;
        HashMap<MethodInfo, MethodCoverage> methods;

        ClassCoverage(String className) {
            this.className = className;
        }

        void setLoaded(ClassInfo ci) {
            if (this.methods == null) {
                this.ci = ci;
                this.covered = true;
                log.info("used class: " + this.className);
                this.methods = new HashMap();
                for (MethodInfo mi : ci.getDeclaredMethodInfos()) {
                    if (mi.isNative() || mi.isAbstract()) continue;
                    MethodCoverage mc = new MethodCoverage(mi);
                    this.methods.put(mi, mc);
                }
            }
        }

        boolean isInterface() {
            if (this.ci == null) {
                this.ci = ClassInfo.getClassInfo(this.className);
            }
            return this.ci.isInterface();
        }

        MethodCoverage getMethodCoverage(MethodInfo mi) {
            if (this.methods == null) {
                this.setLoaded(this.ci);
            }
            return this.methods.get(mi);
        }

        Coverage getCoveredMethods() {
            Coverage cov = new Coverage(0, 0);
            if (this.methods != null) {
                cov.total = this.methods.size();
                for (MethodCoverage mc : this.methods.values()) {
                    if (mc.covered == null) continue;
                    ++cov.covered;
                }
            }
            return cov;
        }

        Coverage getCoveredInsn() {
            Coverage cov = new Coverage(0, 0);
            if (this.methods != null) {
                for (MethodCoverage mc : this.methods.values()) {
                    Coverage c = mc.getCoveredInsn();
                    cov.total += c.total;
                    cov.covered += c.covered;
                }
            }
            return cov;
        }

        Coverage getCoveredBasicBlocks() {
            Coverage cov = new Coverage(0, 0);
            if (this.methods != null) {
                for (MethodCoverage mc : this.methods.values()) {
                    Coverage c = mc.getCoveredBasicBlocks();
                    cov.total += c.total;
                    cov.covered += c.covered;
                }
            }
            return cov;
        }

        Coverage getCoveredBranches() {
            Coverage cov = new Coverage(0, 0);
            if (this.methods != null) {
                for (MethodCoverage mc : this.methods.values()) {
                    Coverage c = mc.getCoveredBranches();
                    cov.total += c.total;
                    cov.covered += c.covered;
                }
            }
            return cov;
        }
    }

    static class MethodCoverage {
        MethodInfo mi;
        BitSet[] covered;
        BitSet basicBlocks;
        BitSet handlers;
        BitSet branches;
        BitSet branchTrue;
        BitSet branchFalse;

        MethodCoverage(MethodInfo mi) {
            this.mi = mi;
            log.info("add method: " + mi.getUniqueName());
        }

        MethodInfo getMethodInfo() {
            return this.mi;
        }

        void setExecuted(ThreadInfo ti, Instruction insn) {
            int idx = ti.getIndex();
            if (this.covered == null) {
                this.covered = new BitSet[idx + 1];
            } else if (idx >= this.covered.length) {
                BitSet[] a = new BitSet[idx + 1];
                System.arraycopy(this.covered, 0, a, 0, this.covered.length);
                this.covered = a;
            }
            if (this.covered[idx] == null) {
                this.covered[idx] = new BitSet(this.mi.getInstructions().length);
            }
            int off = insn.getOffset();
            this.covered[idx].set(off);
            if (showBranchCoverage && insn instanceof IfInstruction) {
                if (this.branchTrue == null) {
                    this.branchTrue = new BitSet(this.mi.getInstructions().length);
                    this.branchFalse = new BitSet(this.branchTrue.size());
                }
                if (!((IfInstruction)insn).getConditionValue()) {
                    this.branchTrue.set(off);
                } else {
                    this.branchFalse.set(off);
                }
            }
        }

        void setCGed(ThreadInfo ti, Instruction insn) {
            BitSet bb = this.getBasicBlocks();
            Instruction next = insn.getNext();
            if (next != null) {
                bb.set(next.getOffset());
            }
        }

        BitSet getExecutedInsn() {
            int nTotal = this.mi.getInstructions().length;
            BitSet bUnion = new BitSet(nTotal);
            if (this.covered != null) {
                for (BitSet b : this.covered) {
                    if (b == null) continue;
                    bUnion.or(b);
                }
            }
            return bUnion;
        }

        Coverage getCoveredInsn() {
            int nTotal = this.mi.getInstructions().length;
            if (excludeHandlers) {
                nTotal -= this.getHandlers().cardinality();
            }
            if (this.covered != null) {
                BitSet bExec = this.getExecutedInsn();
                if (excludeHandlers) {
                    bExec.andNot(this.getHandlers());
                }
                return new Coverage(nTotal, bExec.cardinality());
            }
            return new Coverage(nTotal, 0);
        }

        Coverage getCoveredBranches() {
            BitSet b = this.getBranches();
            int nTotal = b.cardinality();
            int nCovered = 0;
            if (this.branchTrue != null) {
                int n = this.branchTrue.size();
                for (int i = 0; i < n; ++i) {
                    boolean cTrue = this.branchTrue.get(i);
                    boolean cFalse = this.branchFalse.get(i);
                    if (!cTrue || !cFalse) continue;
                    ++nCovered;
                }
            }
            return new Coverage(nTotal, nCovered);
        }

        BitSet getHandlerStarts() {
            BitSet b = new BitSet(this.mi.getInstructions().length);
            ExceptionHandler[] handler = this.mi.getExceptions();
            if (handler != null) {
                for (int i = 0; i < handler.length; ++i) {
                    Instruction hs = this.mi.getInstructionAt(handler[i].getHandler());
                    b.set(hs.getOffset());
                }
            }
            return b;
        }

        BitSet getHandlers() {
            if (this.handlers == null) {
                BitSet hs = this.getHandlerStarts();
                Instruction[] code = this.mi.getInstructions();
                BitSet b = new BitSet(code.length);
                if (!hs.isEmpty()) {
                    for (int i = 0; i < code.length; ++i) {
                        Instruction insn = code[i];
                        if (insn instanceof GOTO) {
                            GOTO gotoInsn = (GOTO)insn;
                            if (gotoInsn.isBackJump() || !hs.get(i + 1)) continue;
                            int handlerEnd = gotoInsn.getTarget().getOffset();
                            ++i;
                            while (i < handlerEnd) {
                                b.set(i);
                                ++i;
                            }
                            continue;
                        }
                        if (!(insn instanceof ReturnInstruction)) continue;
                        ++i;
                        while (i < code.length) {
                            b.set(i);
                            ++i;
                        }
                    }
                }
                this.handlers = b;
            }
            return this.handlers;
        }

        BitSet getBranches() {
            if (this.branches == null) {
                Instruction[] code = this.mi.getInstructions();
                BitSet br = new BitSet(code.length);
                for (int i = 0; i < code.length; ++i) {
                    Instruction insn = code[i];
                    if (!(insn instanceof IfInstruction)) continue;
                    br.set(i);
                }
                this.branches = br;
            }
            return this.branches;
        }

        BitSet getBasicBlocks() {
            if (this.basicBlocks == null) {
                Instruction tgt;
                Instruction[] code = this.mi.getInstructions();
                BitSet bb = new BitSet(code.length);
                bb.set(0);
                for (int i = 0; i < code.length; ++i) {
                    Instruction insn = code[i];
                    if (insn instanceof IfInstruction) {
                        IfInstruction ifInsn = (IfInstruction)insn;
                        Instruction tgt2 = ifInsn.getTarget();
                        bb.set(tgt2.getOffset());
                        tgt2 = ifInsn.getNext();
                        bb.set(tgt2.getOffset());
                        continue;
                    }
                    if (insn instanceof GOTO) {
                        tgt = ((GOTO)insn).getTarget();
                        bb.set(tgt.getOffset());
                        continue;
                    }
                    if (!(insn instanceof InvokeInstruction)) continue;
                    tgt = insn.getNext();
                    bb.set(tgt.getOffset());
                }
                ExceptionHandler[] handlers = this.mi.getExceptions();
                if (handlers != null) {
                    for (int i = 0; i < handlers.length; ++i) {
                        tgt = this.mi.getInstructionAt(handlers[i].getHandler());
                        bb.set(tgt.getOffset());
                    }
                }
                this.basicBlocks = bb;
            }
            return this.basicBlocks;
        }

        Coverage getCoveredBasicBlocks() {
            BitSet bExec = this.getExecutedInsn();
            BitSet bb = this.getBasicBlocks();
            int nCov = 0;
            if (excludeHandlers) {
                BitSet handlers = this.getHandlers();
                bb.and(handlers);
            }
            if (bExec != null) {
                BitSet bCov = new BitSet(bb.size());
                bCov.or(bb);
                bCov.and(bExec);
                nCov = bCov.cardinality();
            }
            return new Coverage(bb.cardinality(), nCov);
        }
    }

    static class Coverage {
        int total;
        int covered;

        Coverage(int total, int covered) {
            this.total = total;
            this.covered = covered;
        }

        public void add(Coverage cov) {
            this.total += cov.total;
            this.covered += cov.covered;
        }

        public int percent() {
            return this.covered * 100 / this.total;
        }

        public String format() {
            String s = this.total > 0 ? String.format("%1$.2f (%2$d/%3$d)", (double)this.covered / (double)this.total, this.covered, this.total) : " -  ";
            return String.format("%1$-18s", s);
        }

        public boolean isPartiallyCovered() {
            return this.covered > 0 && this.covered < this.total;
        }

        public boolean isNotCovered() {
            return this.covered == 0;
        }

        public boolean isFullyCovered() {
            return this.covered == this.total;
        }
    }
}

