/*
 *
 * Behavior Protocols Tools - Parsers, Transformations
 * Copyright (C) 2006-2007  DSRG, Charles University in Prague
 *                          http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */

package org.ow2.dsrg.fm.tbplib.util;


import java.io.FileNotFoundException;
import java.io.OutputStream;


/**
 * Wraps Printstream and provides functions for creating DOT graph descriptions
 * www.graphviz.org
 * @author thorm
 */
public class DOTPrintStream extends java.io.PrintStream {
    
    /**
     * Creates a new instance of DOTPrintStream
     * @param out output stream
     */
    public DOTPrintStream(OutputStream out) {
        super(out);
        println("digraph G {");
    }
    
    protected int component=0;
    
    public void nextComponent(){
        component++;
    }
    
    
    
    /**
     * Creates a new instance of DOTPrintStream
     * @param fileName file name
     * @throws java.io.FileNotFoundException thrown when file is not found
     */
    public DOTPrintStream(String fileName) throws FileNotFoundException {
        super(fileName);
        println("digraph G {");
    }
    
    /**
     * Put line that describes a graph node into the output
     * @param id node identifier
     * @param label node label
     */
    public void node(int id, String label) {
        println("  A_"+component +"_" +id  + nodeOptions(label));
    }

    /**
     * Put line that describes a graph node into the output
     * @param id node identifier
     * @param label node label
     */
    public void node(String id, String label) {
        println("  A_"+component +"_" +id + nodeOptions(label));
    }
    
    /**
     * Put line that describes a graph edge into the output
     * @param from source node identifier
     * @param to target node identifier
     */
    public void edge(int from, int to) {
        println("    " + "A_"+component +"_"+from + " -> " + "A_"+component +"_"+to + ";");
    }
    
    /**
     * Put line that describes a graph edge into the output
     * @param from source node identifier
     * @param to target node identifier
     * @param label edge label
     */
    public void edge(int from, int to, String label) {
        println("    " + "A_"+component +"_"+from + " -> " + "A_"+component +"_"+to + edgeOptions(label));
    }
    
    /**
     * Create string that contains actual node options
     * @param label node name
     * @return string containing node options
     */
    private String nodeOptions(String label) {
        
        return " [label=\"" + label + "\" peripheries=\"" + peripheries + "\""
                + (filled ? " style=filled " : "")
                + ((pos == null) ? "" : (" pos=\"" + pos + "\"")) + "];";
    }
    
    /**
     * Create string that contains actual edge options
     * @param label edge name
     * @return string containing edge options
     */
    private String edgeOptions(String label) {
        if (weight != null) {
            return " [label=\"" + label + "\" weight=\"" + weight + "\"];";
        }
        return " [label=\"" + label + "\"];";
    }
    
    /**
     * Write the last line that finishes the graph declaration and close wrapped
     * stream
     */
    public void DOTclose() {
        println("}");
    }
    
    /**
     * Number of next node peripheries
     */
    public int peripheries = 1;
    
    /**
     * Is the next node filled
     */
    public boolean filled = false;
    
    /**
     * Shape of the next node. Null means default
     */
    String shape = null;
    
    /**
     * Edge weight
     */
    public String weight = null;
    
    /**
     * Next node position
     */
    public String pos = null;
}
