/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved.visitor;

import java.util.ArrayList;
import java.util.List;

import org.ow2.dsrg.fm.tbplib.TBPNode;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAAutomaton;
import org.ow2.dsrg.fm.tbplib.resolved.*;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedEmit;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Translates code from imperative part of TBP from resolved tree into LTSA.
 * 
 * @author caitt3am
 *
 */
public class TranslateImperativeVisitor extends ImperativeVisitor<LTSAAutomaton> {
	
	@Override
	public LTSAAutomaton visitResolvedAssignment(TBPResolvedAssignment assignment) {

		final TBPResolvedValue value = assignment.getValue();
		if(value.isMethodCall()){
			
			TBPResolvedValue nv = new TBPResolvedValue(new LastCallRef(value.getType()));
			TBPResolvedAssignment na = new TBPResolvedAssignment(assignment.getIdf(), nv);
			LTSAAutomaton a = value.getMethodCall().visit(this);
			
			return a.append(LTSAAutomaton.createSimple(na));
		}
		
		return LTSAAutomaton.createSimple(assignment);
	}

	@Override
	public LTSAAutomaton visitResolvedEmit(TBPResolvedEmit resolvedEmit) {
		return LTSAAutomaton.createSimple(resolvedEmit);
	}

    @Override
	public LTSAAutomaton visitResolvedUndefinedEmit(
			TBPResolvedUndefinedEmit resolvedEmit) {
		return LTSAAutomaton.createSimple(resolvedEmit);
	}

	@Override
	public LTSAAutomaton visitResolvedIf(TBPResolvedIf resolvedIf) {
		TBPResolvedCondition condition = resolvedIf.getCondition();
		TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) resolvedIf.getChild(0);
		LTSAAutomaton automaton = code.visit(this);
		LTSAAutomaton automaton2 = null;
				
		if(resolvedIf.hasElse()) {
			TBPResolvedImperativeNode code2 =  (TBPResolvedImperativeNode) resolvedIf.getChild(1);
			automaton2 = code2.visit(this);
		}
		
		return LTSAAutomaton.createChoice(condition, automaton, automaton2);
	}

	@Override
	public LTSAAutomaton visitResolvedImperativeBinaryNode(
			TBPResolvedImperativeBinaryNode node) {
		return null;
	}

	@Override
	public LTSAAutomaton visitResolvedImperativeLeafNode(
			TBPResolvedImperativeLeafNode node) {
		return null;
	}

	@Override
	public LTSAAutomaton visitResolvedImperativeNaryNode(
			TBPResolvedImperativeNaryNode node) {
		return null;
	}

	@Override
	public LTSAAutomaton visitResolvedImperativeNull(
			TBPResolvedImperativeNull resolvedImperativeNull) {
		return LTSAAutomaton.createSimple(resolvedImperativeNull);
	}

	@Override
	public LTSAAutomaton visitResolvedImperativeSequence(
			TBPResolvedImperativeSequence resolvedImperativeSequence) {
		TBPResolvedImperativeNode left = (TBPResolvedImperativeNode) resolvedImperativeSequence.getLeft();
		TBPResolvedImperativeNode right= (TBPResolvedImperativeNode) resolvedImperativeSequence.getRight();
		
		return left.visit(this).append(right.visit(this));
	}

	@Override
	public LTSAAutomaton visitResolvedImperativeUnaryNode(
			TBPResolvedImperativeUnaryNode node) {
		return null;
	}

	private RuntimeException notApplicable(TBPNode node) {
		return new RuntimeException(this.getClass() 
				+ " is not applicable to " + node.getClass());
	}
	

	@Override
	public LTSAAutomaton visitResolvedMangledReaction(
			TBPResolvedMangledReaction node) {
		TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) node.getChild();
		LTSAAutomaton aut = code.visit(this);
		if(node.getBinding().getMethodSignature().getReturnType() == null){
			aut.append(LTSAAutomaton.createSimple(new TBPResolvedReturn(null)));
		}
		return aut;
	}

	@Override
	public LTSAAutomaton visitResolvedReturn(TBPResolvedReturn resolvedReturn) {

		return LTSAAutomaton.createSimple(resolvedReturn);
	}

	@Override
	public LTSAAutomaton visitResolvedSwitch(TBPResolvedSwitch resolvedSwitch) {

		List<TBPResolvedCondition> conditions = resolvedSwitch.getConditions();
		List<LTSAAutomaton> branches = new ArrayList<LTSAAutomaton>(conditions.size());
		
		for(TBPNode node : resolvedSwitch.getChildren()){
			branches.add(((TBPResolvedImperativeNode) node).visit(this));
		}
		
		LTSAAutomaton result = LTSAAutomaton.createMultipleChoice(conditions, branches);
		
		if(!resolvedSwitch.isNondeterministic() && resolvedSwitch.getValue().isMethodCall()){
			LTSAAutomaton automaton = resolvedSwitch.getValue().getMethodCall().visit(this);
			result = automaton.append(result);
		}
		
		return result;
	}

	@Override
	public LTSAAutomaton visitResolvedSync(TBPResolvedSync resolvedSync) {

		TBPResolvedImperativeNode code = (TBPResolvedImperativeNode) resolvedSync.getChild();
		CVRef mutex = resolvedSync.getRef();
		assert mutex.getType().equals(Typedef.MUTEX_TYPE);
		ConstantRef unlock_const = new ConstantRef(Typedef.UNLOCKED, mutex.getType());
		
		TBPResolvedAssignment unlock = new TBPResolvedAssignment(mutex, 
				new TBPResolvedValue(unlock_const));
		
		return LTSAAutomaton.createSimple(mutex)
			.append(code.visit(this))
			.append(LTSAAutomaton.createSimple(unlock));
	}

	@Override
	public LTSAAutomaton visitResolvedThreadContainerNode(
			TBPResolvedThreadContainerNode node) {
		throw notApplicable(node);
	}

	@Override
	public LTSAAutomaton visitResolvedValue(TBPResolvedValue resolvedValue) {
		throw notApplicable(resolvedValue);
	}

	@Override
	public LTSAAutomaton visitResolvedWhile(TBPResolvedWhile resolvedWhile) {
		
		TBPResolvedCondition condition = resolvedWhile.getCondition();
		LTSAAutomaton a = ((TBPResolvedImperativeNode) resolvedWhile.getChild()).visit(this);

		return LTSAAutomaton.createWhile(condition, a);
	}

}
