/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved.visitor;

import java.io.PrintWriter;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.ow2.dsrg.fm.tbplib.BinaryNode;
import org.ow2.dsrg.fm.tbplib.TBPNode;
import org.ow2.dsrg.fm.tbplib.UnaryNode;
import org.ow2.dsrg.fm.tbplib.resolved.*;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedAcceptRequest;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedEmit;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedEmitResponse;
import org.ow2.dsrg.fm.tbplib.resolved.util.MethodSignature;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Prints parts of resolved hierarchy into given stream (System.out is default).
 * Use {@link #printComponentSpecification(ResolvedComponentSpecification)} to
 * print whole component.
 * 
 * @author caitt3am
 *
 */
public class PrintResolvedComponentVisitor implements TBPResolvedVisitor<Object> {

	private PrintWriter o;
	private int indent = 0;
	
	private static final int indent_size = 4;
	private static final String opening = " {";
	private static final String closing = " }";
	
	public PrintResolvedComponentVisitor() {
		o = new PrintWriter(System.out, true);
	}
	
	
	public PrintResolvedComponentVisitor(PrintWriter writer){
		o = writer;
	}
	
	private void indent(){
		for(int i = 0; i < indent; i++)
			o.append(' ');
	}
	
	private void increaseIndent() {
		indent += indent_size;
	}
	
	private void decreaseIndent() {
		indent -= indent_size;
		assert indent >= 0;
	}
	
	private void open() {
		increaseIndent();
		println(opening);		
	}
	
	private void close() {
		decreaseIndent();
		println(closing);
	}
	
	private void println(String s) {
		o.println(s);
		indent();
	}
	
	private void appendSection(String name, Collection<? extends TBPResolvedNode> container){
		o.append(name);
		open();
		for (TBPResolvedNode node : container) {
			node.visit(this);
		}
		close();
	}
	
	private void appendType(Typedef t) {
		o.append(t.getName());
		o.append(opening);
		Iterator<String> it = t.getEnums().iterator();
		if(it.hasNext()){
			o.append(it.next());
		}
		while(it.hasNext()){
			o.append(", ");
			o.append(it.next());
		}
		println(closing);
	}
	
	private void appendCondition(TBPResolvedCondition c) {
		o.append(c.toString());
	}
	
	public void printComponentSpecification(ResolvedComponentSpecification rcs){
		o.write(rcs.getComponentname());
		open();
		
			o.append("types");
			open();
				for (Typedef type : rcs.getTypes()) {
					appendType(type);
				}
			close();
			
			o.append("vars");
			open();
				for(TBPResolvedVardef v : rcs.getVardefs().values()){
					println(v.toString());
				}
			close();

			appendSection("provisions", rcs.getProvisions());
			appendSection("reactions", rcs.getReactions());
			appendSection("definitions", rcs.getDefinitions().values());
			appendSection("threads", rcs.getThreads());

		close();
	}

	@Override
	public Object visitResolvedProvisionContainerNode(
			TBPResolvedProvisionContainerNode resolvedContainer) {
		o.append(resolvedContainer.getName());
		appendBlock(resolvedContainer.getChild());
		return null;
	}
	
	@Override
	public Object visitResolvedMangledReaction(TBPResolvedMangledReaction node) {
		o.append(node.getBinding().toString());
		appendBlock(node.getChild());
		return null;
	}

	@Override
	public Object visitResolvedThreadContainerNode(
			TBPResolvedThreadContainerNode node) {
		o.append(node.getName());
		appendBlock(node.getChild());
		return null;
	}
	
	@Override
	public Object visitResolvedMethodDeclaration(
			TBPResolvedMethodDefinition resolvedMethodDeclaration) {
		MethodSignature signature = resolvedMethodDeclaration.getMethodSignature();
		o.print(signature);
		appendBlock(resolvedMethodDeclaration.getChild());
		return null;
	}
	
	
	private void appendBinary(BinaryNode node, String middle) {
		o.append('(');
		((TBPResolvedNode)node.getLeft()).visit(this);
		o.append(')');
		o.append(middle);
		o.append('(');
		((TBPResolvedNode)node.getRight()).visit(this);
		o.append(')');
	}
	
	private void appendUnary(UnaryNode node, String postfix) {
		o.append('(');
		((TBPResolvedNode)node.getChild()).visit(this);
		o.append(')').append(postfix);
	}

	
	@Override
	public Object visitResolvedAlternative(TBPResolvedAlternative resolvedAlternative) {
		appendBinary(resolvedAlternative, "+");
		return null;
	}
	
	@Override
	public Object visitResolvedParallel(TBPResolvedParallel resolvedParallel) {
		appendBinary(resolvedParallel, "|");
		return null;
	}

	@Override
	public Object visitResolvedParallelOr(
			TBPResolvedParallelOr resolvedParallelOr) {
		appendBinary(resolvedParallelOr, "||");
		return null;
	}
	
	@Override
	public Object visitResolvedProvisionSequence(
			TBPResolvedProvisionSequence resolvedSequence) {
		appendBinary(resolvedSequence, ";");
		return null;
	}
	
	@Override
	public Object visitResolvedRepetition(
			TBPResolvedRepetition resolvedRepetition) {
		appendUnary(resolvedRepetition, "*");
		return null;
	}

	@Override
	public Object visitResolvedLimitedReentrancy(TBPResolvedLimitedReentrancy node) {
		appendUnary(node, "|");
		o.append(Integer.toString(node.getLimit()));
		return null;
	}

	@Override
	public Object visitResolvedProvisionNull(
			TBPResolvedProvisionNull resolvedProvisionNull) {
		o.append("NULL");
		return null;
	}

	@Override
	public Object visitResolvedAssignment(
			TBPResolvedAssignment resolvedAssignment) {
		o.append(resolvedAssignment.getIdf().getName());
		o.append(" <- ");
		resolvedAssignment.getValue().visit(this);
		return null;
	}

	@Override
	public Object visitResolvedEmitResponse(
			TBPResolvedEmitResponse resolvedEmitResponse) {
		o.append('!');
		o.append(Integer.toString(resolvedEmitResponse.getEventIndex()));
		o.append('$');
		return null;
	}

	@Override
	public Object visitResolvedAcceptRequest(
			TBPResolvedAcceptRequest resolvedAcceptRequest) {
		o.append('?');
		o.append(Integer.toString(resolvedAcceptRequest.getEventIndex()));
		o.append('^');
		return null;
	}
	
	@Override
	public Object visitResolvedIf(TBPResolvedIf resolvedIf) {
		o.append("if(");
		appendCondition(resolvedIf.getCondition());
		o.append(')');
		open();
		((TBPResolvedNode)resolvedIf.getChild(0)).visit(this);
		close();
		if(resolvedIf.getChildCount()>1){
			o.append(" else ");
			open();
			((TBPResolvedNode)resolvedIf.getChild(1)).visit(this);
			close();
		}
		return null;
	}

	@Override
	public Object visitResolvedImperativeBinaryNode(TBPResolvedImperativeBinaryNode node) {
		return null;
	}

	@Override
	public Object visitResolvedImperativeLeafNode(TBPResolvedImperativeLeafNode node) {
		return null;
	}

	@Override
	public Object visitResolvedImperativeNaryNode(TBPResolvedImperativeNaryNode node) {
		return null;
	}

	@Override
	public Object visitResolvedImperativeNull(TBPResolvedImperativeNull resolvedImperativeNull) {
		o.append("NULL");
		return null;
	}

	@Override
	public Object visitResolvedImperativeSequence(TBPResolvedImperativeSequence resolvedImperativeSequence) {
		TBPResolvedImperativeNode left = (TBPResolvedImperativeNode) resolvedImperativeSequence.getLeft();
		TBPResolvedImperativeNode right = (TBPResolvedImperativeNode) resolvedImperativeSequence.getRight();
		
		left.visit(this);
		println(";");
		right.visit(this);
		
		return null;
	}

	@Override
	public Object visitResolvedImperativeUnaryNode(TBPResolvedImperativeUnaryNode node) {
		return null;
	}
	
	@Override
	public Object visitResolvedProvisionBinaryNode(TBPResolvedProvisionBinaryNode node) {
		return null;
	}

	@Override
	public Object visitResolvedProvisionLeafNode(
			TBPResolvedProvisionLeafNode node) {
		return null;
	}

	@Override
	public Object visitResolvedProvisionNaryNode(
			TBPResolvedProvisionNaryNode node) {
		return null;
	}

	@Override
	public Object visitResolvedProvisionUnaryNode(
			TBPResolvedProvisionUnaryNode node) {
		return null;
	}

	@Override
	public Object visitResolvedReturn(TBPResolvedReturn resolvedReturn) {
		o.append("return ");
		o.append(resolvedReturn.getReturnValue().getName());
		return null;
	}

	@Override
	public Object visitResolvedSwitch(TBPResolvedSwitch swit) {
		List<TBPNode> branches = swit.getChildren();
		List<ConstantRef> cases = swit.getCases();
		TBPResolvedImperativeNode db = swit.getDefaultBranch();
		
		if(swit.isNondeterministic()){
			println("switch(?) {");
			for(TBPNode b : branches){
				TBPResolvedNode branch = (TBPResolvedNode) b;
				increaseIndent();
				println("case: ");
				
				branch.visit(this);
				decreaseIndent();
				println("");
			}
			
			return null;
		}
		
		o.append("switch(");
		swit.getValue().visit(this);
		println(") {");
		for (int i = 0; i < cases.size(); i++) {
			String c = cases.get(i).getName();
			TBPResolvedImperativeNode b = (TBPResolvedImperativeNode) branches.get(i);
			o.append(c);
			increaseIndent();
			println(": ");
			
			b.visit(this);
			decreaseIndent();
			println("");
		}
		if(db != null){
			o.append("default: ");
			increaseIndent();
			db.visit(this);
			decreaseIndent();
			println("");
		}
		
		return null;
	}

	@Override
	public Object visitResolvedSync(TBPResolvedSync resolvedSync) {
		o.append("sync(").append(resolvedSync.getRef().getName()).append(')');
		appendBlock(resolvedSync.getChild());
		return null;
	}


	@Override
	public Object visitResolvedValue(TBPResolvedValue resolvedValue) {
		if(resolvedValue.isReference()){
			o.append(resolvedValue.getReference().getName());
		} else {
			resolvedValue.getMethodCall().visit(this);
		}
		return null;
	}

	@Override
	public Object visitResolvedWhile(TBPResolvedWhile resolvedWhile) {
		o.append("while(");
		appendCondition(resolvedWhile.getCondition());
		o.append(")");
		appendBlock(resolvedWhile.getChild());
		return null;
	}

	private void appendBlock(TBPNode block) {
		open();
			((TBPResolvedNode)block).visit(this);
		close();
	}


	@Override
	public Object visitResolvedEmit(TBPResolvedEmit resolvedEmit) {
		o.append("!").append(resolvedEmit.toString());
		return null;
	}
	
       @Override
	public Object visitResolvedUndefinedEmit(TBPResolvedUndefinedEmit resolvedEmit) {
		o.append(resolvedEmit.toString());
		return null;
	}

}
