/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved;

import java.util.List;

import org.ow2.dsrg.fm.tbplib.VardefException;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedVardef;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * @author caitt3am
 *
 */
public class TBPResolvedVardef implements Cloneable {
	
	private boolean mutex;
	private String name;
	private Typedef type;
	private String init;
	
	public TBPResolvedVardef(TBPParsedVardef def, List<Typedef> types) {
		this.name = def.getName();
		this.mutex = def.isMutex();
		
		if(!mutex){
			for(Typedef type : types){
				if(type.getName().equals(def.getTypename())){
					this.type = type;
					for(String en : type.getEnums()){
						if(en.equals(def.getInitialValue())){
							this.init = en;
						}
					}
				}
			}
			if(type == null)
				throw new VardefException("Unknown type referenced");
			if(init == null)
				throw new VardefException("Initial value " + def.getInitialValue() 
						+ " not found in type " + type.getName());
		} else {
			this.type = Typedef.MUTEX_TYPE;
			this.init = Typedef.UNLOCKED;
		}
		
	}
	
	public String getName() {
		return name;
	}
	
	public void prefix(String prefix){
		name = prefix + name; 
	}
	
	public boolean isMutex() {
		return mutex;
	}
	
	public Typedef getType() {
		return type;
	}
	
	public String getInitialValue() {
		return init;
	}
	
	@Override
	public String toString() {
		if(mutex){
			return "mutex " + name;
		} else {
			return type.getName() + " " + name + " = " + init;
		}
	}
	
	@Override
	protected TBPResolvedVardef clone()  {
		try {
			return (TBPResolvedVardef) super.clone();
		} catch (CloneNotSupportedException e) {
			throw new RuntimeException("Cloning TBPResolvedVardef failed.");
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof TBPResolvedVardef))
			return false;
		final TBPResolvedVardef other = (TBPResolvedVardef) obj;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;
		return true;
	}
}
