/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.ow2.dsrg.fm.tbplib.TBPNode;
import org.ow2.dsrg.fm.tbplib.TBPResolvingException;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedImperativeNode;
import org.ow2.dsrg.fm.tbplib.resolved.visitor.TBPResolvedVisitor;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Switch statement. Branches are kept as children of this node (including default branch). 
 * Case labels are kept as list of reference to constants.  
 * 
 * @author caitt3am
 *
 */
public class TBPResolvedSwitch extends TBPResolvedImperativeNaryNode {

	private final List<ConstantRef> cases;
	private final TBPResolvedImperativeNode defaultBranch;
	private final TBPResolvedValue value;
	
	/**
	 * Constructs non-deterministic switch. Non-deterministic switch has no value,
	 * no case labels and no default branch. It has only branches. 
	 * @param branches 
	 */
	public TBPResolvedSwitch(List<TBPResolvedImperativeNode> branches) {
		this.cases = null;
		this.defaultBranch = null;
		this.value = null;
		for(TBPResolvedImperativeNode branch : branches){
			addChild(branch);
		}
	}
	
	/**
	 * Constructs deterministic switch, with optional default branch. 
	 * @param value - over what we switch
	 * @param branches - nodes representing branches  
	 * @param cases - labels for branches
	 * Branches and cases with same index correspond to each other. The number of branches 
	 * can be greater by one than the number of cases - in that case the last branch is 
	 * default branch.
	 * @throws TBPResolvingException if there is type mismatch between switch value and case
	 * labels.
	 */
	public TBPResolvedSwitch(
			TBPResolvedValue value,
			List<TBPResolvedImperativeNode> branches,
			List<String> cases) {
		
		final int csize = cases.size();
		final int bsize = branches.size();
		if(! (csize == bsize || bsize==csize+1)){
			throw new TBPResolvingException("Number of labels and cases does not match") ;
		}
		
		this.cases = new ArrayList<ConstantRef>(csize);
		final Typedef type = value.getType();
		for(String s : cases){
			if(!type.getEnums().contains(s))
				throw new TBPResolvingException("Label " + s + " is not part of type " + type.getName());
			this.cases.add(new ConstantRef(s,type));
		}
		if(bsize-1 == csize){
			this.defaultBranch = branches.get(branches.size()-1);
		} else {
			this.defaultBranch = null;
		}
		this.value = value;
		for(TBPResolvedImperativeNode branch : branches){
			assert branch != null;
			addChild(branch);
		}
	}
	
	/**
	 * Copy constructor used to make (deep) copy of switch statement. Arguments are rearanged
	 * to allow overloading.
	 * 
 	 * @param cases - list of case labels (in form of constant references not strings)
	 * @param value - over what we switch
	 * @param branches - nodes representing branches  
	 */
	public TBPResolvedSwitch(List<ConstantRef> cases,
			TBPResolvedValue value,
			List<TBPResolvedImperativeNode> branches
			){
		final int csize = cases.size();
		final int bsize = branches.size();
		if(! (csize == bsize || bsize==csize+1)){
			throw new TBPResolvingException("Number of labels and cases does not match") ;
		}
		if(bsize-1 == csize){
			this.defaultBranch = branches.get(branches.size()-1);
		} else {
			this.defaultBranch = null;
		}
		
		this.value = value;
		this.cases = cases;
		for(TBPNode b : branches){
			addChild(b);
		}		
	}

	public List<ConstantRef> getCases() {
		return cases;
	}
	
	public TBPResolvedImperativeNode getDefaultBranch() {
		return defaultBranch;
	}
	
	/**
	 * Returns value used to select between branches.
	 * <code>null</code> if switch is non-deterministic.
	 */
	public TBPResolvedValue getValue() {
		return value;
	}

	public boolean isNondeterministic(){
		return cases == null;
	}

	/**
	 * For each branch return condition that must be satisfied to 
	 * enter that branch.
	 */
	public List<TBPResolvedCondition> getConditions()
	{
		final int n = getChildCount();
		if(isNondeterministic()){
			return Collections.nCopies(n, new TBPResolvedCondition());	
		}
		
		List<TBPResolvedCondition> result = new ArrayList<TBPResolvedCondition>(n);
		Reference ref = (value.isReference()) ? value.getReference() : new LastCallRef(value.getType());
		for(ConstantRef s : cases){
			result.add(new TBPResolvedCondition(ref, s));
		}
		if(defaultBranch != null){
			result.add(new TBPResolvedCondition(ref, cases));
		}
		assert result.size() == getChildCount();
		return result;
	}
	
	@Override
	public <E> E visit(TBPResolvedVisitor<E> visitor) {
		return visitor.visitResolvedSwitch(this);
	}
}
