/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.resolved;

import java.util.List;

import org.ow2.dsrg.fm.tbplib.TBPResolvingException;


/**
 * Typesafe condition - value object. It has three types: 
 * <ul><li>comparison of constant and variable</li>
 *     <li>'?'</li>
 *     <li>comparison of variable not being in set of constants</li>
 * </ul>
 * <br>
 * Note: Immutable class.
 * 
 * @author caitt3am
 *
 */
public class TBPResolvedCondition implements Cloneable {

	private boolean negate;
	private final Reference left;
	private final ConstantRef right;
	private final List<ConstantRef> right_list;
	private final List<CVRef> left_list;
	private final ConditionType condition_type;
	
	/**
	 * Represents types of condition that {@link TBPResolvedCondition} can hold.
	 */
	 public static enum ConditionType {
		 CONDITION_VARIABLE_CONSTANT, /// Represents standard comparison variable (left) == CONSTANT (right)
		CONDITION_VARIABLE_IN_CONSTANTS, /// Represents condition that variable (left) is in given list (right_list). Note only used in negated form is not in right_list. Used for default branch in switch.
		CONDITION_ANY, /// Use to to represent "?" in condition. It means no conditions.
		//TODO Meaning (Check if meaning correct)
		CONDITION_MULTIPLE_VARIABLES_TO_CONSTANT_ONE_BY_ONE /// Simulates multiple CONDITION_VARIABLE_CONSTANT. i-th variable in variable list (left_list) is checked if equals i-th constant in constant list (right_list). If all holds true, otherwise false. Used for inlining functions. 
	}

	/**
	 * Constructs comparison condition. One of sides must be constant. 
	 * @param left - left side of comparison.
	 * @param right - right side of comparison. 
	 */
	public TBPResolvedCondition(Reference left, Reference right) {
		this.right_list = null;
		this.left_list = null;
		this.negate = false;
		this.condition_type = ConditionType.CONDITION_VARIABLE_CONSTANT;
		assert left != null && right != null;

		if(!left.getType().equals(right.getType())){
			throw new TBPResolvingException("Type mismatch between " 
					+ left.getName() + " and " + right.getName());
		}
		if (left instanceof ConstantRef) {
			this.right = (ConstantRef) left;
			this.left = right;
		} else {
			
			if(!(right instanceof ConstantRef)){
				throw new TBPResolvingException("Condition must have constant in it");
			}
			
			this.left = left;
			this.right = (ConstantRef) right;			
		}

	}

	/**
	 * Constructs 'not in' condition that is true when given variable is not in list of values. 
	 * This type of condition cannot be created directly in source language and should be used
	 * in automatons only where it is used to translate default branch of switch statement. 
	 * @param ref - value tested
	 * @param list - forbidden values
	 */
	public TBPResolvedCondition(Reference ref, List<ConstantRef> list){
		this.left = ref;
		this.left_list = null;
		this.right = null;
		this.right_list = list;
		this.negate = true;
		this.condition_type = ConditionType.CONDITION_VARIABLE_IN_CONSTANTS;
	}
	
	/**
	 * Creates condition for set of comparisons. This type of condition
	 * is needed in automatons. It does not occur in resolved tree.
	 * @param varList - list of variables to compare
	 * @param constList - list of constant to compare with
	 */
	public TBPResolvedCondition(List<CVRef> varList, List<ConstantRef> constList){
		this.left = null;
		this.right= null;
		this.right_list = constList;
		this.left_list  = varList;
		this.negate = false;
		this.condition_type = ConditionType.CONDITION_MULTIPLE_VARIABLES_TO_CONSTANT_ONE_BY_ONE;
	}
	/**
	 * Constructs '?' condition that is always true (even negated)
	 */
	public TBPResolvedCondition() {
		this.left = this.right = null;
		this.right_list = null;
		this.left_list = null;
		this.negate = false;
		this.condition_type = ConditionType.CONDITION_ANY;
	}
	
	// Getters
	public final ConditionType getConditionType() {
		return condition_type;
	}
	
	public final List<CVRef> getLeftList() {
		return left_list;
	}
	
	public final List<ConstantRef> getRightList() {
		return right_list;
	}
	
	public final Reference getLeft() {
		return left;
	}
	
	public final ConstantRef getRight() {
		return right;
	}
	
	public final boolean isNegated() {
		return negate;
	}

	
	public boolean isNondeterministic() {
		return left == null && right_list == null;
	}
	
	public boolean isDeterministic(){
		return !isNondeterministic();
	}
	
	public TBPResolvedCondition createNegation() {
		TBPResolvedCondition result = clone();
		result.negate = !negate;
		return result;
	}

	@Override
	public TBPResolvedCondition clone() {
		try {
			return (TBPResolvedCondition) super.clone();
		} catch (CloneNotSupportedException e) {
			throw new RuntimeException("Cloning TBPResolvedCondition failed");
		}
	}
	
	@Override
	public String toString() {
		if(isNondeterministic())
			return "?";
		if(left_list != null){
			
			assert right_list != null;
			assert right_list.size() == left_list.size();
			
			final int s = right_list.size() - 1;
			StringBuilder bld = new StringBuilder("(");
			for(int i = 0; i < s; i++)
				bld.append(left_list.get(i).getName()).append(" == ")
					.append(right_list.get(i).getName()).append(" && ");
			bld.append(left_list.get(s).getName()).append(" == ").append(
					right_list.get(s).getName());
			
			return bld.append(")").toString();
			
		} else if(right_list == null){
			if(negate)
				return left.getName() + " != " + right.getName();
			else
				return left.getName() + " == " + right.getName();
		} else {
			final int s = right_list.size() - 1;
			StringBuilder bld = new StringBuilder(left.getName());
			if(negate)
				bld.append(" not");
			bld.append(" in (");
			
			for(int i = 0; i < s; i++)
				bld.append(right_list.get(i).getName()).append(", ");
			bld.append(right_list.get(s).getName()).append(")");
			return bld.toString();
		}
	}
}

