/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.parsed.visitor;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.ow2.dsrg.fm.tbplib.TBPNode;
import org.ow2.dsrg.fm.tbplib.TBPResolvingException;
import org.ow2.dsrg.fm.tbplib.TBPUndefinedEmitException;
import org.ow2.dsrg.fm.tbplib.parsed.MethodCall;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedAssignment;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedCondition;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedEmit;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedIf;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeNull;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedImperativeSequence;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedMethodDeclaration;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedReturn;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSwitch;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedSync;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedThreadContainerNode;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedValue;
import org.ow2.dsrg.fm.tbplib.parsed.TBPParsedWhile;
import org.ow2.dsrg.fm.tbplib.resolved.CVRef;
import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.Reference;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedAssignment;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedCondition;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedIf;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedImperativeNode;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedImperativeNull;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedImperativeSequence;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedMethodDefinition;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedReturn;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedSwitch;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedSync;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedThreadContainerNode;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedUndefinedEmit;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedValue;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedVardef;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedWhile;
import org.ow2.dsrg.fm.tbplib.resolved.events.TBPResolvedEmit;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;
import org.ow2.dsrg.fm.tbplib.resolved.util.MethodSignature;
import org.ow2.dsrg.fm.tbplib.resolved.util.UtilClass;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Translates trees of parsed imperative nodes into resolved imperative nodes.
 * 
 * @author caitt3am
 *
 */
public class ImperativeResolvingVisitor extends TBPParsedCheckingVisitor<TBPResolvedImperativeNode>{

	private Map<String,TBPResolvedVardef> vardefs;
	private List<Typedef> types;
	private Map<String, MethodSignature> sigs;
	
	// in dynamic scope of visiting method declaration this 
	// field is set to its signature
	private MethodSignature actualMethodSignature;
	
	/**
	 * Creates visitor that transforms parsing nodes to resolve nodes in threads 
	 * and reaction sections
	 * 
	 * @param vardefs accessible variable
	 * @param types list of accessible types
	 * @param sigs signatures of accessible declared method
	 */
	public ImperativeResolvingVisitor(Map<String, TBPResolvedVardef> vardefs,
			List<Typedef> types,
			 Map<String, MethodSignature> sigs) {
		this.vardefs = vardefs;
		this.types = types;
		this.sigs = sigs;
	}

	@Override
	public TBPResolvedImperativeNode visitParsedSwitch(TBPParsedSwitch node) {
		TBPResolvedImperativeNode result;
		
		List<TBPResolvedImperativeNode> b = new ArrayList<TBPResolvedImperativeNode>();
		for (TBPNode ch : node.getChildren()) {
			TBPParsedImperativeNode n = (TBPParsedImperativeNode) ch;
			b.add(n.visit(this));
		}
		
		
		if(node.isNondeterministic()){
			result = new TBPResolvedSwitch(b);
		} else {
			try {
				TBPResolvedValue rval = (TBPResolvedValue) node.getValue().visit(this);  
				result = new TBPResolvedSwitch(rval, b, node.getCases());
			} catch (TBPUndefinedEmitException e){
				return new TBPResolvedUndefinedEmit(e.getMethodCall());
			}
		}
		result.setAnnotation(node.getAnnotation());
		return result;
	}
	
	@Override
	public TBPResolvedImperativeNode visitParsedIf(TBPParsedIf node) {
		
		TBPResolvedImperativeNode result;
		TBPResolvedCondition rcond = resolveCondition(node.getCondition());
		TBPParsedImperativeNode ifb = (TBPParsedImperativeNode) node.getChild(0);
		TBPResolvedImperativeNode rifb = ifb.visit(this);
		
		if(node.getChildCount() > 1){
			TBPParsedImperativeNode elseb = (TBPParsedImperativeNode) node.getChild(1);
			result = new TBPResolvedIf(rcond, rifb, elseb.visit(this)) ;
		} else {
			result = new TBPResolvedIf(rcond, rifb, null);
		}
		result.setAnnotation(node.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitParsedWhile(TBPParsedWhile node) {
		TBPResolvedImperativeNode result;
		TBPParsedNode block = (TBPParsedNode) node.getChild();
		result = new TBPResolvedWhile(
				resolveCondition(node.getCondition()), 
				block.visit(this));
		result.setAnnotation(node.getAnnotation());
		return result;
	}


	@Override
	public TBPResolvedImperativeNode visitParsedImperativeSequence(
			TBPParsedImperativeSequence node) {
		TBPResolvedImperativeNode result;
		TBPParsedImperativeNode left = (TBPParsedImperativeNode) node.getLeft();
		TBPParsedImperativeNode right = (TBPParsedImperativeNode) node.getRight();
		result = new TBPResolvedImperativeSequence(left.visit(this), right.visit(this));
		result.setAnnotation(node.getAnnotation());
		return result;
	}
	
	@Override
	public TBPResolvedImperativeNode visitParsedAssignment(
			TBPParsedAssignment node) {
		
		TBPResolvedImperativeNode result;
		TBPParsedValue child = (TBPParsedValue) node.getChild();

		Reference ref = UtilClass.makeReference(node.getIdf(), types, vardefs, actualMethodSignature);
		if (ref instanceof CVRef) {
			CVRef cref = (CVRef) ref;
			try {
				TBPResolvedValue rval = (TBPResolvedValue) visitParsedValue(child);
				result = new TBPResolvedAssignment(cref, rval);
				result.setAnnotation(node.getAnnotation());
				return result;
			}catch(TBPUndefinedEmitException e){
				return new TBPResolvedUndefinedEmit(e.getMethodCall());
			}
		}
		throw new TBPResolvingException("Only component variables can be assigned!");
		
	}

	// Translate condition; it is not method from Visitor interface
	public TBPResolvedCondition resolveCondition(TBPParsedCondition node) {

		if(node.getLeft() != null){
			Reference lref = UtilClass.makeReference(node.getLeft(), types, vardefs, actualMethodSignature);
			Reference rref = UtilClass.makeReference(node.getRight(), types, vardefs, actualMethodSignature);
			return new TBPResolvedCondition(lref, rref);
		} else {
			return new TBPResolvedCondition();
		}
	}



	@Override
	public TBPResolvedImperativeNode visitParsedEmit(TBPParsedEmit node) {
		TBPResolvedImperativeNode result;
		try {
			
			Binding binding = UtilClass.makeBinding(sigs, node.getMethodCall(), 
					types, vardefs, actualMethodSignature);		
			result = new TBPResolvedEmit(binding);
			
		} catch (TBPUndefinedEmitException e){
			result = new TBPResolvedUndefinedEmit(e.getMethodCall());
		}
		result.setAnnotation(node.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitParsedImperativeNull(
			TBPParsedImperativeNull node) {
		TBPResolvedImperativeNode result = new TBPResolvedImperativeNull();
		result.setAnnotation(node.getAnnotation());
		return result;
	}


	@Override
	public TBPResolvedImperativeNode visitParsedMethodDeclaration(
			TBPParsedMethodDeclaration node) {
		
		MethodSignature ms = sigs.get(node.getFullname());
		assert ms != null : "This declaration " + node.getFullname() + "was not processed!";

		actualMethodSignature = ms;
		
			TBPParsedImperativeNode code = (TBPParsedImperativeNode) node.getChild();			
			TBPResolvedImperativeNode result = new TBPResolvedMethodDefinition(
					ms, code.visit(this));
		
		actualMethodSignature = null;
		
		result.setAnnotation(node.getAnnotation());
		return result;
	}


	@Override
	public TBPResolvedImperativeNode visitParsedReturn(TBPParsedReturn node) {
		TBPResolvedImperativeNode result;		
		final String idf = node.getIdf();
		final ConstantRef const_ref = UtilClass.makeConstantReference(idf, types);
		
		if(actualMethodSignature == null){
			throw new TBPResolvingException("Cannot use return outside reaction");
		}
		
		if(!const_ref.getType().equals(actualMethodSignature.getReturnType())){
			throw new TBPResolvingException("Type mismatch: bad return type (in "
					 + actualMethodSignature.toString() + ")");
		}
		
		result = new TBPResolvedReturn(const_ref);
		result.setAnnotation(node.getAnnotation());
		return result;
	}


	@Override
	public TBPResolvedImperativeNode visitParsedSync(TBPParsedSync node) {
		TBPResolvedImperativeNode result;
		CVRef ref = UtilClass.makeCVReference(node.getMutex(), vardefs);
		TBPParsedNode parsedCode = (TBPParsedNode) node.getChild();
		
		result = new TBPResolvedSync(ref,	parsedCode.visit(this));
		result.setAnnotation(node.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitParsedValue(TBPParsedValue node) {
		TBPResolvedImperativeNode result;
		MethodCall value = node.getMethodCall();
		String varname = node.getVarname();
		
		assert value != null || varname != null;
		
		if(value==null){
			Reference ref = UtilClass.makeReference(varname, types, vardefs, actualMethodSignature);
			result = new TBPResolvedValue(ref);
		} else { 
			Binding bindMethodSignature = UtilClass.makeBinding(sigs, 
					value, types, vardefs,	actualMethodSignature);			
			result = new TBPResolvedValue( new TBPResolvedEmit(bindMethodSignature));
		}
		result.setAnnotation(node.getAnnotation());
		return result;
	}

	@Override
	public TBPResolvedImperativeNode visitParsedThreadContainerNode(
			TBPParsedThreadContainerNode node) {
		
		TBPParsedImperativeNode code = (TBPParsedImperativeNode) node.getChild();
		return new TBPResolvedThreadContainerNode(node.getName(), code.visit(this));
	}

}