/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */

package org.ow2.dsrg.fm.tbplib.parsed;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.ow2.dsrg.fm.tbplib.EventTable;
import org.ow2.dsrg.fm.tbplib.TBPNode;
import org.ow2.dsrg.fm.tbplib.TBPResolvingException;
import org.ow2.dsrg.fm.tbplib.parsed.visitor.ImperativeResolvingVisitor;
import org.ow2.dsrg.fm.tbplib.parsed.visitor.ProvisionResolvingVisitor;
import org.ow2.dsrg.fm.tbplib.parsed.visitor.TBPParsedDefaultVisitor;
import org.ow2.dsrg.fm.tbplib.parsed.visitor.TBPParsedVisitor;
import org.ow2.dsrg.fm.tbplib.resolved.ResolvedComponentSpecification;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedImperativeNode;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedMethodDefinition;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedProvisionNode;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedThreadContainerNode;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedVardef;
import org.ow2.dsrg.fm.tbplib.resolved.util.MethodSignature;
import org.ow2.dsrg.fm.tbplib.resolved.util.UtilClass;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * Holder for parsed TBP component specification.
 * 
 * @author caitt3am
 *
 */
public class ParsedComponentSpecification {

	private String componentname;
    private List<Typedef>  types;
    private List<TBPParsedVardef> vardefs;
    private List<TBPParsedProvisionContainerNode> provisions;
    private List<String> provision_names;
    private List<TBPParsedImperativeNode> reactions;
    private List<TBPParsedThreadContainerNode> threads;

    public ParsedComponentSpecification(String componentname, 
			List<TBPParsedProvisionContainerNode> provisions, 
			List<TBPParsedImperativeNode> reactions,
			List<TBPParsedThreadContainerNode> threads,
			List<Typedef> types, 
			List<TBPParsedVardef> vardefs) {

		this.componentname = componentname;
		this.provisions = provisions;
		this.threads = threads;
		this.reactions = reactions;
		this.types = types;
		this.vardefs = vardefs;
		
		provision_names = new ArrayList<String>();
		
		TBPParsedVisitor<Object> collect_names = new TBPParsedDefaultVisitor<Object>(){
			@Override
			public
			Object visitParsedAccept(TBPParsedAccept n) {
				provision_names.add(n.getFullname());
				return null;
			}
		};
		
		for(TBPParsedProvisionNode provision: provisions){
			Iterator<TBPNode> iterator = provision.iterator();
			while(iterator.hasNext()){
				TBPParsedProvisionNode n = (TBPParsedProvisionNode) iterator.next();
				n.visit(collect_names);
			}
		}
	}
    
    /**
     * All occurrences of unlimited reentrancy in all provisions are replaced 
     * by limited reentrancy. This metod changes provisions in place. It does not
     * make copies.
     *  
     * @param limit parameter of limited reentrancy
     */
    public void limitReentrancy(final int limit){
    	TBPParsedVisitor<Object> replaceReentrancy = new TBPParsedDefaultVisitor<Object>(){
    		@Override
    		public Object visitUnlimitedReentrancy(TBPUnlimitedReentrancy node) {
    			
    			TBPParsedProvisionNode child = (TBPParsedProvisionNode) node.getChild();
    			TBPNode parent = node.getParent();
    			int idx = parent.getChildren().indexOf(node);
    			
    			node.setChild(new TBPParsedProvisionNull());
    			TBPNode replacement = new TBPLimitedReentrancy(child, limit);
    			parent.setChild(idx, replacement);
    			
    			return null;
    		}
    		
    	};
    	
    	for(TBPParsedProvisionNode provision: provisions){
			for(TBPNode node : provision){
				((TBPParsedProvisionNode)node).visit(replaceReentrancy);
			}
		}
    }
    
    private Map<String, TBPResolvedVardef> resolveVardefs(){
    	Map<String, TBPResolvedVardef> rdefs = new HashMap<String, TBPResolvedVardef>(vardefs.size());

		for (TBPParsedVardef def : vardefs) {
			TBPResolvedVardef duplicate;
			duplicate = rdefs.put(def.getName(), new TBPResolvedVardef(def,types));
			if(duplicate != null){
				throw new TBPResolvingException("Duplicit variable name " 
						+ duplicate.getName());
			}
		}
		
		return rdefs;
    }
    
    private Map<String, MethodSignature> resolveSignatures() {
    	return UtilClass.generateMethodSignatures(getReactions(), types);
	}
    
	public ResolvedComponentSpecification resolve(EventTable table){
		
		Map<String, TBPResolvedVardef> rdefs = resolveVardefs();
		Map<String, MethodSignature> sigs = resolveSignatures();
		
		// Resolving provion part
		List<TBPResolvedProvisionNode> resolved_provisions = new ArrayList<TBPResolvedProvisionNode>();
		TBPParsedVisitor<TBPResolvedProvisionNode> pvisitor = 
			new ProvisionResolvingVisitor(types, rdefs, sigs, table);
		
		for (TBPParsedProvisionNode pnode : provisions) {
			resolved_provisions.add(pnode.visit(pvisitor));
		}
		
		// Resolving threads
		List<TBPResolvedThreadContainerNode> resolved_threads = new ArrayList<TBPResolvedThreadContainerNode>();
		TBPParsedVisitor<TBPResolvedImperativeNode> ivisitor = 
			new ImperativeResolvingVisitor(rdefs, types, sigs);
		
		for (TBPParsedThreadContainerNode imperativeNode : threads) {
			resolved_threads.add(
					(TBPResolvedThreadContainerNode)imperativeNode.visit(ivisitor));
		}
		
		List<TBPResolvedMethodDefinition> resolved_reactions = new ArrayList<TBPResolvedMethodDefinition>();
		for (TBPParsedImperativeNode imperativeNode : reactions) {
			resolved_reactions.add((TBPResolvedMethodDefinition) imperativeNode.visit(ivisitor));
		}
		
		HashMap<String, TBPResolvedMethodDefinition> declarations = new HashMap<String, TBPResolvedMethodDefinition>();
		
		for (TBPResolvedMethodDefinition decl : resolved_reactions) {
			declarations.put(decl.getMethodSignature().getFullname(), decl);
		}
		
		return new ResolvedComponentSpecification(componentname, types, rdefs, 
				resolved_provisions, resolved_threads, declarations);
    }

	public List<Typedef> getTypes() {
		return types;
	}

	public List<TBPParsedVardef> getVardefs() {
		return vardefs;
	}
	
	public List<String> getVarnames(){
		List<String> result = new ArrayList<String>(vardefs.size());
		for(TBPParsedVardef vardef : vardefs){
			result.add(vardef.getName());
		}
		return result;
	}

	public List<TBPParsedProvisionContainerNode> getProvisions() {
		return provisions;
	}

	/**
	 * Names of all provisions referenced in provisions section.
	 */
	public List<String> getProvisionNames() {
		return provision_names;
	}

	public List<TBPParsedThreadContainerNode> getThreads() {
		return threads;
	}
	
	public List<TBPParsedImperativeNode> getReactions() {
		return reactions;
	}
	
	public String getName(){
		return componentname;
	}

}
