/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.parsed;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.ow2.dsrg.fm.tbplib.TBPNode;
import org.ow2.dsrg.fm.tbplib.parsed.visitor.PrefixingVisitor;
import org.ow2.dsrg.fm.tbplib.util.Typedef;

/**
 * This class represents composition of components.
 * 
 * @author caitt3am
 *
 */
public class ParsedComponentComposition {
	List<ParsedComponentSpecification> specs = new ArrayList<ParsedComponentSpecification>();
	
	public void addComponent(ParsedComponentSpecification spec) {
		specs.add(spec);
	}
	
	/**
	 * Creates new component composed from components 
	 * added through {@link #addComponent(ParsedComponentSpecification)}.
	 * Parts of components are put together. Component variables are prefixed
	 * with names of component to avoid name clashes. If two types are the same
	 * they are merged into one. Other name clashes results in throwing 
	 * an exception. 
	 * <p> 
	 * The added components are consumed during the process.
	 * 
	 * @param name - Name of the new component
	 * @return parsed component specification
	 * @throws ParsedException in case of name clashes
	 */
	public ParsedComponentSpecification getComposition(String name) 
		throws ParsedException {

		List<String> names = new ArrayList<String>();
		List<TBPParsedProvisionContainerNode> provisions = new ArrayList<TBPParsedProvisionContainerNode>(); 
		List<TBPParsedImperativeNode> reactions = new ArrayList<TBPParsedImperativeNode>();
		List<TBPParsedThreadContainerNode> threads = new ArrayList<TBPParsedThreadContainerNode>();
		List<Typedef> types = new ArrayList<Typedef>();
		List<TBPParsedVardef> vardefs = new ArrayList<TBPParsedVardef>();
		
		for(ParsedComponentSpecification spec : specs){
			names.add(spec.getName());
			provisions.addAll(spec.getProvisions());
			reactions.addAll(spec.getReactions());
			threads.addAll(spec.getThreads());
			types.addAll(spec.getTypes());
			vardefs.addAll(spec.getVardefs());			
		}

		Set<String> n = new TreeSet<String>();
		n.addAll(names);
		if( n.size() != names.size() ){
			throw new ParsedException("Duplicit component names");
		}
		
		n.clear();
		for(TBPParsedProvisionContainerNode p : provisions){
			n.add(p.getName());
		}
		if(n.size() != provisions.size()){
			throw new ParsedException("Duplicit provision names");
		}
		
		n.clear();
		for(TBPParsedImperativeNode r : reactions){
			TBPParsedMethodDeclaration decl = (TBPParsedMethodDeclaration) r;
			n.add(decl.getFullname());
		}
		if(n.size() != reactions.size()){
			throw new ParsedException("Duplicit reactions");
		}
		
		Collections.sort(types, new Comparator<Typedef>(){

			public int compare(Typedef o1, Typedef o2) {
				return o1.getName().compareTo(o2.getName());
			}
			
		});

		List<Typedef> merged_types = new ArrayList<Typedef>();
		final int s = types.size() - 1;
		for(int i = 0; i < s; i++){
			Typedef first = types.get(i);
			Typedef second = types.get(i+1);
			if(first.getName().equals(second.getName())){
				if(!first.getEnums().equals(second.getEnums())){
					throw new ParsedException("Redefinition of " + first.getName());
				}
			} else {
				merged_types.add(first);
			}
		}
		if(s>=0)
			merged_types.add(types.get(s));
		
		for(ParsedComponentSpecification psc : specs){
			prefixComponentVars(psc);
		}
		
		return new ParsedComponentSpecification(name, provisions, reactions, 
					threads, merged_types, vardefs);
	}
	
	private static void prefixComponentVars(ParsedComponentSpecification psc) {
		final String prefix = psc.getName() + ":";
		final PrefixingVisitor visitor = new PrefixingVisitor(prefix,
				psc.getVarnames());

		for(TBPParsedImperativeNode r : psc.getReactions()){
			for(TBPNode n : r){
				TBPParsedNode node = (TBPParsedNode) n;
				node.visit(visitor);
			}
		}
		
		for(TBPParsedThreadContainerNode r : psc.getThreads()){
			for(TBPNode n : r){
				TBPParsedNode node = (TBPParsedNode) n;
				node.visit(visitor);
			}
		}
		 
		for(TBPParsedVardef vardef : psc.getVardefs()){
			vardef.setName(prefix + vardef.getName());
		}
	}
}
