/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.ltsa;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Node of a graph. It knows outgoing edges a has mark field to be used 
 * by walking algorithms.
 * 
 * @see Utils#processGraph(State, EdgeVisitor)
 * 
 * @author caitt3am
 *
 */
public class State {

	private List<Edge> edges;
	private boolean mark;
	
	public State() {
		edges = new ArrayList<Edge>();
	}
	

	/**
	 * Add edge into list of outgoing edges but does not change 
	 * its source field.
	 */
	public void addEdge(Edge e) {
		assert e.getSource() == null;
		edges.add(e);
	}

	/**
	 * Creates new edge and adds it to list of outgoing edges.
	 * @param data - payload of new edge
	 * @param target - second end of new edge
	 */
	public void addEdge(Object data, State target) {
		edges.add(new Edge(data, this, target));
	}
	
	/**
	 * Creates new edge with no payload and adds it to list of outgoing edges.
	 * @param target - second end of new edge
	 * @see #addEdge(Object, State)
	 */
	public void addNullEdge(State target){
		edges.add(new Edge(null, this, target));
	}
	
	/**
	 * Removes edge from its list. Does not change edge itself (source field 
	 * will still point to this node).
	 * @param e - what to remove
	 */
	public void removeEdge(Edge e) {
		boolean remove = edges.remove(e);
		assert remove;
	}
	
	public void addEdges(Collection<? extends Edge> c) {
		edges.addAll(c);
	}
	
	public List<Edge> getEdges() {
		return edges;
	}
	
	public void Mark() {
		assert mark != true;
		mark = true;
	}
	
	public void Unmark() {
		assert mark;
		mark = false;
	}
	
	public boolean isMarked() {
		return mark;
	}
	
	@Override
	public String toString() {
		StringBuilder bld = new StringBuilder(State.class.getSimpleName() + '$' + toHex(this.hashCode()) + " Edges(");
		
		final int s = edges.size() - 1;
		for(int i=0; i<s; i++){
			//bld.append(edges.get(i).getSymbol()).append(", ");
			bld.append(toHex(edges.get(i).getTarget().hashCode()));
			bld.append(", ");
		}
		if(s >= 0)
			//bld.append(edges.get(s).getSymbol());
			bld.append(toHex(edges.get(s).getTarget().hashCode()));
		
		bld.append(")");
		return bld.toString();
	}
	
	/** Converts given parameter into string that holds hexadecimal representation. 
	 * String has always length 8. Spaces are added to the end.
	 * 
	 * @param hashCode Number to convert
	 * @return * character long string with hex code of parameter
	 */
	public static String toHex(int hashCode) {
		StringBuffer sb = new StringBuffer(8);
		sb.append(Integer.toHexString(hashCode));
		for(int i = sb.length(); i < 8; i++) {
			sb.append(' ');
		}
		return sb.toString();
	}
}
