/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.ltsa;

import java.util.*;
import java.util.Map.Entry;


/**
 * Nondeterministic automaton representing provision. Contains 
 * methods to create new automatons and {@link #determinize()}
 * to convert this to deterministic automaton.
 * 
 * @author caitt3am
 *
 */
public class NondetermisticAutomaton extends AutomatonImpl {
	private static final boolean DEBUG = false;
	
	private NondetermisticAutomaton(List<State> states, State start, Set<State> final_states) {
		super(states, start, final_states);
	}
	
	private void set(NondetermisticAutomaton a){
		this.start = a.start;
		this.final_states = a.final_states;
		this.states = a.states;
	}
	
	public NondetermisticAutomaton addAlternative(NondetermisticAutomaton aut) {
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addAlternative(aut= " + aut + ")\n"); System.out.println("This=" + this + '\n');}
		
		assert Collections.disjoint(states, aut.states);
		start.addEdges(aut.start.getEdges());
		
		final_states.addAll(((NondetermisticAutomaton) aut).final_states);
		if(((NondetermisticAutomaton) aut).final_states.contains(((NondetermisticAutomaton) aut).start)){
			final_states.add(start);
		}
		
		states.addAll(((NondetermisticAutomaton) aut).states);
		
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addAlternative - result(=this) = " + this); }
		return this;
	}

	private static NondetermisticAutomaton createParallel(NondetermisticAutomaton a, NondetermisticAutomaton b){
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":createParallel(a= " + a + ", b=" + b + ")\n"); }
		final int s1 = a.states.size();
		final int s2 = b.states.size();
		final int f1 = a.final_states.size();
		final int f2 = b.final_states.size();

		State temp[][] = new State[s1][s2];
		
		for(int i = 0; i<s1; i++){
			for(int j = 0; j < s2; j++){
				temp[i][j] = new State();
			}
		}
		
		// not fastest version
		for(int i = 0; i < s1; i++){
			for(int j = 0; j < s2; j++){
				State s = temp[i][j];
				
				// first coordinate
				State c1 = a.states.get(i);
				for(Edge e : c1.getEdges()){
					int idx = a.states.indexOf(e.getTarget());
					s.addEdge(new Edge(e.getSymbol(), temp[idx][j]));
				}
				
				// second coordinate
				State c2 = b.states.get(j);
				for(Edge e : c2.getEdges()){
					int idx = b.states.indexOf(e.getTarget());
					s.addEdge(new Edge(e.getSymbol(), temp[i][idx]));
				}
			}
		}
		
		State new_start = temp[a.states.indexOf(a.start)][b.states.indexOf(b.start)];		
		Set<State> new_final_states = new HashSet<State>();
		List<Integer> finals1 = new ArrayList<Integer>(f1);
		List<Integer> finals2 = new ArrayList<Integer>(f2);
		
		for (State s : a.final_states) {
			finals1.add(a.states.indexOf(s));
		}

		for (State s : b.final_states) {
			finals2.add(b.states.indexOf(s));
		}
		
		for(Integer i : finals1)
			for(Integer j : finals2)
				new_final_states.add(temp[i][j]);
		
		
		List<State> new_states = new ArrayList<State>(s1 * s2);

		for(int i = 0; i < s1; i++){
			for(int j = 0; j < s2; j++){
				new_states.add(temp[i][j]);
			}
		}
		
		NondetermisticAutomaton result = new NondetermisticAutomaton(new_states, new_start, new_final_states);
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":createParallel - result= " + result + "\n"); }
		
		return result; 
	}
	
	
	public NondetermisticAutomaton addParallel(NondetermisticAutomaton aut) {
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addParllel(...)\n");}
		set(createParallel(this, aut));
		return this;
	}

	
	public NondetermisticAutomaton addParallelOr(NondetermisticAutomaton aut) {
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addParallelOr(...)\n");}
		// TODO Spatny zpusob ... takto by vysledek s line codes byl spatny (nebyl tereministicky)
		NondetermisticAutomaton par = createParallel(this, aut);
		addAlternative(aut).addAlternative(par);
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addParallelOr - result - see last addAlternative result\n");}
		return this;
	}

	
	public NondetermisticAutomaton addReentrancy(int k) {
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addReentrancy(k = " + k + ")\nThis = " +this + "\n");}
		// A || A || A ... is equivalent to A + A|A + A|A|A ...
		assert k >= 1;
		if(k==1)
			return this;
		
		List<NondetermisticAutomaton> nfs = new ArrayList<NondetermisticAutomaton>(k-1);
		nfs.add(createParallel(this, this));
		
		for(int i = 0; i < k-2; i++){
			nfs.add(createParallel(this, nfs.get(i)));
		}
		
		for (NondetermisticAutomaton impl : nfs) {
			addAlternative(impl);
		}
		
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addReentrancy result=" + this + "\n");}
		return this;
	}

	
	public NondetermisticAutomaton addRepetition() {
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addRepetition()\nThis=" + this + "\n");}
		
		State new_start = new State();
		final_states.add(new_start);
		for (State s : final_states) {
			s.addEdges(start.getEdges());
		}
		start = new_start;
		states.add(new_start);
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addRepetition result=" + this + "\n");}
		return this;
	}

	
	public NondetermisticAutomaton addSequence(NondetermisticAutomaton aut) {
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addSequence(aut=" + aut + ")\nThis=" + this + "\n");}
		assert Collections.disjoint(states, aut.states);
		
		for(State s : final_states){		
			s.addEdges(aut.start.getEdges());
		}	
		
		if(aut.final_states.contains(aut.start)){
			final_states.addAll(aut.final_states);
		} else {
			final_states = aut.final_states;
		}
		
		states.addAll(aut.states);
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":addSequence result=" + this + "\n");}
		return this;
	}


	public static NondetermisticAutomaton createNull() {
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":createNull()\n");}
		// Note: We remove static NULL automaton, because this automaton has 
		//         unmodifiable state and final_states list ... we need modifiable automation 
		//         for automatons operation 
			State       start = new State();
			List<State> states = new ArrayList<State>(1);
			states.add(start);
			Set<State>  final_states = new HashSet<State>();
			final_states.add(start);
			
		return new NondetermisticAutomaton(states, start, final_states);
	}

	public static NondetermisticAutomaton createSimple(int symbol) {
		if (DEBUG) { System.out.println(NondetermisticAutomaton.class.getSimpleName() + ":createSimplel(symbol=" + symbol + ")\n");}
		
		State start = new State();
		State fin = new State();
		start.addEdge(new Edge(symbol,fin));
		
		List<State> states = new ArrayList<State>(2);
		states.add(start);
		states.add(fin);
		
		HashSet<State> finals = new HashSet<State>();
		finals.add(fin);
		
		return new NondetermisticAutomaton(states, start, finals);
	}

	
	public DeterministicAutomaton determinize() {

		Set<State> det_final_states = new HashSet<State>();
		List<State> det_states = new ArrayList<State>();
		Queue<State> queue = new ArrayDeque<State>();
		Map<Collection<State>, State> set_to_state = 
			new HashMap<Collection<State>, State>();
		Map<State, Collection<State>> state_to_set =
			new HashMap<State, Collection<State>>();
		
		State det_start = new State();
		List<State> start_set = Collections.singletonList(start);
		
		queue.add(det_start);
		state_to_set.put(det_start, start_set);
		set_to_state.put(start_set, det_start);
		
		State s;
		Collection<State> act_set;
		while( (s=queue.poll()) != null){
			if (DEBUG) {
				System.out.println("    Determinize stats: queueLen:" + queue.size() + ", currentState: " + s);
				// Calc statistics of graph we creating
				final int cnt_states = state_to_set.size();
				int cnt_edges = 0;
				for(State s1 : state_to_set.keySet()) {
					cnt_edges += s1.getEdges().size();
				}
				System.out.println("      temp graph: states:" + cnt_states + ", edges:" + cnt_edges);
			}

			det_states.add(s);
			act_set = state_to_set.get(s);
			
			// get accesible sets
			EdgeMultiMap m = new EdgeMultiMap();
			for (State state : act_set) {
				for (Edge edge : state.getEdges()) {
					m.put(edge.getSymbol(), edge.getTarget());
				}
			}
			
			// each entry results in one edge in new graph
			for (Entry<Integer, List<State>> entry : m.entrySet()) {
				int symbol = entry.getKey();
				List<State> set_target = entry.getValue();
				
				// get deterministic state for target
				State target = set_to_state.get(set_target);
				if(target == null){ // we have to create one
					
					target = new State();
					
					if(!Collections.disjoint(set_target, final_states)){
						det_final_states.add(target);						
					}
					
					queue.add(target);
					set_to_state.put(set_target, target);
					state_to_set.put(target, set_target);
				}
				
				s.addEdge(new Edge(symbol, target));
			}
			
		}
		
		return new DeterministicAutomaton(det_states, det_start, det_final_states);
	}

	private static class EdgeMultiMap extends HashMap<Integer, List<State>> {

		private static final long serialVersionUID = 1L;

		public void put(Integer key, State value) {

			if(containsKey(key)){
				get(key).add(value);
			} else {
				List<State> new_bucket = new ArrayList<State>(1);
				new_bucket.add(value);
				put(key, new_bucket);
			}
		}
	}

}
	