/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib.ltsa;

import java.util.*;
import java.util.Map.Entry;


/**
 * @author caitt3am
 *
 */
public class DeterministicAutomaton extends AutomatonImpl {

	public DeterministicAutomaton(List<State> states, State start_state,
			Set<State> final_states) {
		super(states, start_state, final_states);
	}
	
	public DeterministicAutomaton makeMinimal(){

		Map<State, Integer> state_to_class = new HashMap<State, Integer>();
		Map<Integer, List<State>> class_to_state = new HashMap<Integer, List<State>>();
		
		createPartition(states, final_states, class_to_state, state_to_class);

		Set<Integer> new_states = class_to_state.keySet();
		
		Map<State, Map<Integer, Integer>> neigborhood = getStatesNeigborhood(states, state_to_class);
		
		int num_states = new_states.size();
		List<State> new_s = new ArrayList<State>(num_states);
		Map<Integer, Integer> classes_to_representants = 
			new HashMap<Integer, Integer>();
		
		int r = 0;
		for(Integer i : new_states){
			new_s.add(new State());
			classes_to_representants.put(i, r++);
		}
		
		for(int c : new_states){
			State source = new_s.get(classes_to_representants.get(c));
			Map<Integer,Integer> edges = neigborhood.get(class_to_state.get(c).get(0));
			for(Entry<Integer, Integer> entry : edges.entrySet()) {
				final Integer key = entry.getKey();
				final Integer value = entry.getValue();
				final Integer c1 = classes_to_representants.get(value);
				final State target = new_s.get(c1);
				source.addEdge(new Edge(key.intValue(), target));
			}
		}
		
		final Integer cl = state_to_class.get(start);
		final Integer repr1 = classes_to_representants.get(cl);
		State new_start = new_s.get(repr1);
		
		Set<State> new_finals = new HashSet<State>();
		for(State f : final_states){
			Integer c = state_to_class.get(f);
			Integer repr = classes_to_representants.get(c);
			State new_f = new_s.get(repr);
			new_finals.add(new_f);
		}
		return new DeterministicAutomaton(new_s, new_start, new_finals);
	}
	
	/**
	 * Returns mapping from state to mapping from symbol to target 
	 * class (determined from state_to_class mapping)
	 * @param states
	 * @param state_to_class
	 * @return
	 */
	private static Map<State, Map<Integer, Integer>> getStatesNeigborhood(
			List<State> states, Map<State,Integer> state_to_class) {
		
		Map<State, Map<Integer, Integer>> env = new HashMap<State, Map<Integer,Integer>>();
		
		for(State s : states){
			Map<Integer, Integer> my_env = new TreeMap<Integer, Integer>();
			for(Edge e : s.getEdges()){
				my_env.put(e.getSymbol(), state_to_class.get(e.getTarget()));
			}
			env.put(s, my_env);
		}
		
		return env;
	}
	
	private static void createPartition(
			List<State> all_states, Collection<State> final_states,
			Map<Integer, List<State>> class_to_state, Map<State, Integer> state_to_class) {
		
		// just to be sure
		class_to_state.clear();
		state_to_class.clear();
		
		List<Integer> classes = new ArrayList<Integer>();
		
		for(State s : all_states){
			state_to_class.put(s, 0);
		}
		for(State s : final_states){
			state_to_class.put(s, 1);
		}
		
		class_to_state.put(0, new ArrayList<State>());
		class_to_state.put(1, new ArrayList<State>());
		
		for(State s : all_states){
			class_to_state.get(state_to_class.get(s)).add(s);
		}
		
		classes.add(0);
		classes.add(1);
		int last_class = 2;
		
		boolean has_changed = true;
		while(has_changed){

			has_changed = false;
			Map<State, Map<Integer, Integer>> env = getStatesNeigborhood(all_states, state_to_class);
			List<Integer> removeList = new ArrayList<Integer>();
			List<Integer> addList = new ArrayList<Integer>();
			
			for(Integer c : classes){
				List<State> list = class_to_state.get(c);
				Set<Map<Integer, Integer>> ss = new HashSet<Map<Integer, Integer>>();
				for(State s : list){
					ss.add(env.get(s));
				}
				int new_cl = ss.size();

				if(new_cl == 1){
					continue;
				}
				
				// split class c into new_cl new classes
				// update classes, class_to_state, state_to_class
				removeList.add(c); // deferring removal to avoid modification of collection during iteration
				
				// preparing for new classes
				for(int i = 0; i < new_cl; i++){
					int key = last_class+i;
					addList.add(key);
					class_to_state.put(key, new ArrayList<State>());
				}
				
				List<Map<Integer, Integer>> trans_list =
					new ArrayList<Map<Integer,Integer>>(ss);
				for(State s : list){
					int cls_idx =  trans_list.indexOf(env.get(s));
					assert cls_idx != -1;
					int new_class = last_class + cls_idx;
					state_to_class.put(s, new_class);
					class_to_state.get(new_class).add(s);
				}
				class_to_state.remove(c);
				last_class += new_cl;
				has_changed = true;
			}
			classes.removeAll(removeList);	
			classes.addAll(addList);
		}
	}
		 
}
