/*
 *
 * Threaded Behavior Protocols  - Parsers, Transformations
 * Copyright (C) 2008   DSRG, Charles University in Prague
 *                      http://dsrg.mff.cuni.cz/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA
 *
 */
package org.ow2.dsrg.fm.tbplib;

import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.ow2.dsrg.fm.tbplib.resolved.ConstantRef;
import org.ow2.dsrg.fm.tbplib.resolved.Reference;
import org.ow2.dsrg.fm.tbplib.resolved.TBPResolvedVardef;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;

/**
 * Simple implementation of event table. Each type of event has its own 
 * class with overridden equals() and hashCode(). They are stored together
 * in pair of hash tables that hold translations from event to code
 * and vice versa.
 * 
 * @author caitt3am
 *
 */
public class EventTableImpl implements EventTable {

	private int freeId;
	private Map<Event, Integer> event_to_code;
	private Map<Integer, Event> code_to_event;

	public EventTableImpl() {
		freeId = 0;
		event_to_code = new HashMap<Event, Integer>();
		code_to_event = new HashMap<Integer, Event>();
	}
	
	private int getEventCode(Event event){
		Integer code = event_to_code.get(event);
		if(code == null){
			code = assignFreeId();
			event_to_code.put(event, code);
			code_to_event.put(code, event);
		}
		return code;
	}

	private final int assignFreeId(){
		assert freeId != Integer.MAX_VALUE;
		return freeId++;
	}
	

	
	@Override
	public int assignIdToAssignment(TBPResolvedVardef to, String value) {
		Event event = new AssignmentEvent(to.getName(), value);		
		return getEventCode(event);
	}

	@Override
	public int assignIdToRequest(Binding ms) {
		Event event = new RequestEvent(ms);
		return getEventCode(event);
	}

	@Override
	public int assignIdToResponse(String fullname, ConstantRef ref) {
		Event event = new ResponseEvent(fullname, (ref==null)? null : ref.getName());
		return getEventCode(event);
	}

	@Override
	public EventType getType(int code) {
		Event event = code_to_event.get(code);
		if(event == null){
			throw new EventTableException("Unassigned code");
		}
		return event.getType(code);
	}
	
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder("Event table:\n");
		for(Entry<Integer, Event> entry : code_to_event.entrySet())
			sb.append(entry.getKey()).append(" - ").append(entry.getValue()).append("\n");
		return sb.toString();
	}
	
	public Event getEvent(int code){
		return code_to_event.get(code);
	}
	

	public abstract static class Event {

		public abstract EventType getType(int code);
	}
	
	public static class RequestEvent extends Event {

		private final List<String> names;
		private final String fullname;
		private final int hashCode;
		
		public RequestEvent(Binding binding) {
			this.fullname = binding.getMethodSignature().getFullname();
			this.names = new ArrayList<String>(binding.getValues().size());
			for(Reference ref : binding.getValues()){
				assert ref instanceof ConstantRef;
				names.add(ref.getName());
			}
			this.hashCode = names.hashCode() + fullname.hashCode();
		}
		
		@Override
		public EventType getType(int code) {
			return EventType.REQUEST;
		}

		@Override
		public int hashCode() {
			return hashCode;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			final RequestEvent other = (RequestEvent) obj;
			if (!names.equals(other.names) || !fullname.equals(other.fullname))
				return false;
			return true;
		}
		
		@Override
		public String toString() {
			StringBuilder sb = new StringBuilder(fullname);
			for(String name : names){
				sb.append(":");
				sb.append(name);
			}
			return sb.toString();
		}
	}
	
	public static class ResponseEvent extends Event {
		
		private final String fullname;
		private final String retValue;
		
		public ResponseEvent(String fullname, String retValue) {
			assert fullname != null;
			this.fullname = fullname;
			this.retValue = retValue;
			
		}

		@Override
		public EventType getType(int code) {
			return EventType.RESPONSE;
		}
		
		public String getFullname() {
			return fullname;
		}
		
		public String getRetValue() {
			return retValue;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result
					+ ((fullname == null) ? 0 : fullname.hashCode());
			result = prime * result
					+ ((retValue == null) ? 0 : retValue.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (!(obj instanceof ResponseEvent))
				return false;
			final ResponseEvent other = (ResponseEvent) obj;
			if (!fullname.equals(other.fullname))
				return false;
			if (retValue == null) {
				if (other.retValue != null)
					return false;
			} else if (!retValue.equals(other.retValue))
				return false;
			return true;
		}
		
		@Override
		public String toString() {
			return fullname + ": " + retValue;
		}
		
	}
	
	public static class AssignmentEvent extends Event {
		String lval, rval;

		public AssignmentEvent(String lval, String rval) {
			assert lval != null && rval != null;
			this.lval = lval;
			this.rval = rval;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + lval.hashCode();
			result = prime * result + rval.hashCode();
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			final AssignmentEvent other = (AssignmentEvent) obj;
			if (!lval.equals(other.lval))
				return false;
			if (!rval.equals(other.rval))
				return false;
			return true;
		}

		@Override
		public String toString() {
			return lval + " <- " + rval;
		}
		@Override
		public EventType getType(int code) {
			return EventType.ASSIGNMENT;
		}
		
		
		
	}
}
