package de.uka.ipd.sdq.workflow.pcm.configurations;

import java.util.ArrayList;
import java.util.List;

import de.uka.ipd.sdq.workflow.launchconfig.ConstantsContainer;
import de.uka.ipd.sdq.workflow.pcm.runconfig.SensitivityAnalysisConfiguration;

/**
 * Base class of workflow configuration objects where the corresponding workflow
 * transforms PCM models into some kind of code
 * 
 * @author Steffen Becker
 * 
 */
public abstract class AbstractCodeGenerationWorkflowRunConfiguration extends
		AbstractPCMWorkflowRunConfiguration {

	public enum CodeGenerationAdvice {
		SIMULATION("simulation_template_methods"), POJO("pojo_template_methods"), EJB3(
				"ejb3_template_methods"), PROTO("prototype_template_methods");

		private String templateFile;

		CodeGenerationAdvice(String templateFile) {
			this.templateFile = templateFile;
		}

		public String getTemplateFile() {
			return templateFile;
		}

	}

	protected boolean cleanupCode = false;
	protected boolean overwriteWithoutAsking = false;
	protected String pluginID = ConstantsContainer.PLUGIN_ID;
	protected boolean loadMiddlewareAndCompletionFiles = false;
	protected CodeGenerationAdvice codeGenerationAdvice = CodeGenerationAdvice.SIMULATION;

	// The sensitivity analysis configurations have been moved here because the
	// PCM2CodeJob
	// requires the Sensitivity Analysis Config anyways. Also for other code
	// generation purposes (e.g. Protocom), sensitivity analysis can make sense.
	protected List<SensitivityAnalysisConfiguration> sensitivityAnalysisConfigurationList = new ArrayList<SensitivityAnalysisConfiguration>();
	protected boolean sensitivityAnalysisEnabled;

	/**
	 * @return Returns whether the generated code should be deleted at the end
	 *         of the workflow run
	 */
	public boolean shouldDeleteGeneratedCodeOnCleanup() {
		return cleanupCode;
	}

	/**
	 * Sets whether the generated code should be deleted at the end of the
	 * workflow
	 * 
	 * @param cleanupCode
	 *            True if the generated code should be deleted at the end of the
	 *            workflow
	 */
	public void setCleanupCode(boolean cleanupCode) {
		checkFixed();
		this.cleanupCode = cleanupCode;
	}

	/**
	 * @return Returns the ID of the Eclipse plugin to be generated by the code
	 *         transformation. This is the name of the Eclipse project which
	 *         will contain the generated code
	 */
	public String getPluginID() {
		return sensitivityAnalysisEnabled ? pluginID
				+ getSensitivityAnalysisIdentifier() : pluginID;
	}

	private String getSensitivityAnalysisIdentifier() {
		String result = "";
		for (SensitivityAnalysisConfiguration sac : sensitivityAnalysisConfigurationList) {
			result += "." + sac.getRunNo();
		}
		return result;
	}

	/**
	 * Sets the name of the project which will contain the generated code
	 * 
	 * @param pluginID
	 *            The name of the project containing the generated code
	 */
	public void setPluginID(String pluginID) {
		checkFixed();
		this.pluginID = pluginID;
	}

	/**
	 * Set whether the code generation may overwrite any existing artifacts
	 * without asking the user. If set to false, the code generation has to ask
	 * the user first.
	 * 
	 * @param overwriteWithoutAsking
	 */
	public void setOverwriteWithoutAsking(boolean overwriteWithoutAsking) {
		this.overwriteWithoutAsking = overwriteWithoutAsking;
	}

	/**
	 * Get whether the code generation may overwrite any existing artifacts
	 * without asking the user. If set to false, the code generation has to ask
	 * the user first.
	 * 
	 * @return true if the code can be overwritten without asking, else false
	 */
	public boolean isOverwriteWithoutAsking() {
		return overwriteWithoutAsking;
	}

	/**
	 * Set whether the middleware files and completion files should be loaded,
	 * too. This is for example required for the simulation of linking
	 * resources.
	 * 
	 * @param loadMiddlewareAndCompletionFiles
	 *            the loadMiddlewareAndCompletionFiles to set
	 */
	public void setLoadMiddlewareAndCompletionFiles(
			boolean loadMiddlewareAndCompletionFiles) {
		this.loadMiddlewareAndCompletionFiles = loadMiddlewareAndCompletionFiles;
	}

	/**
	 * Returns whether the middleware files and completion files should be
	 * loaded, too. This is for example required for the simulation of linking
	 * resources.
	 * 
	 * @return the loadMiddlewareAndCompletionFiles
	 */
	public boolean isLoadMiddlewareAndCompletionFiles() {
		return loadMiddlewareAndCompletionFiles;
	}

	public boolean isSensitivityAnalysisEnabled() {
		return sensitivityAnalysisEnabled;
	}

	public List<SensitivityAnalysisConfiguration> getSensitivityAnalysisConfigurations() {
		if (!isSensitivityAnalysisEnabled())
			throw new UnsupportedOperationException(
					"GetSensitivityAnalysisConfiguration is only supported if isSensitivityAnaysisEnabled is true!");

		return sensitivityAnalysisConfigurationList;
	}

	public void setSensitivityAnalysisEnabled(boolean sensitivityAnalysisEnabled) {
		checkFixed();
		this.sensitivityAnalysisEnabled = sensitivityAnalysisEnabled;
	}

	public void setSensitivityAnalysisConfiguration(
			SensitivityAnalysisConfiguration sensitivityConfig) {
		checkFixed();
		this.sensitivityAnalysisConfigurationList = new ArrayList<SensitivityAnalysisConfiguration>();
		this.sensitivityAnalysisConfigurationList.add(sensitivityConfig);
	}

	public void setSensitivityAnalysisConfigurationList(
			List<SensitivityAnalysisConfiguration> sensitivityConfigList) {
		checkFixed();
		this.sensitivityAnalysisConfigurationList = sensitivityConfigList;

	}

	public String getCodeGenerationAdvicesFile() {
		return this.codeGenerationAdvice.getTemplateFile();
	}

	public void setCodeGenerationAdvicesFile(CodeGenerationAdvice advice) {
		this.codeGenerationAdvice = advice;
	}

}
