package edu.kit.ipd.sdq.kamp.test;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.util.List;

import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.compare.diff.metamodel.DiffElement;
import org.eclipse.emf.compare.diff.metamodel.DiffModel;
import org.eclipse.emf.compare.diff.metamodel.DifferenceKind;
import org.eclipse.emf.compare.diff.metamodel.ModelElementChangeLeftTarget;
import org.eclipse.emf.compare.diff.metamodel.ModelElementChangeRightTarget;
import org.eclipse.emf.compare.diff.metamodel.UpdateReference;
import org.eclipse.emf.compare.diff.service.DiffService;
import org.eclipse.emf.compare.match.metamodel.MatchModel;
import org.eclipse.emf.compare.match.service.MatchService;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import de.uka.ipd.sdq.pcm.core.composition.AssemblyConnector;
import de.uka.ipd.sdq.pcm.repository.BasicComponent;
import de.uka.ipd.sdq.pcm.repository.OperationInterface;
import de.uka.ipd.sdq.pcm.repository.OperationProvidedRole;
import de.uka.ipd.sdq.pcm.repository.OperationRequiredRole;
import de.uka.ipd.sdq.pcm.repository.ProvidedRole;
import de.uka.ipd.sdq.pcm.repository.RepositoryComponent;
import de.uka.ipd.sdq.pcm.repository.RequiredRole;
import edu.kit.ipd.sdq.kamp.core.Activity;
import edu.kit.ipd.sdq.kamp.core.ActivityElementType;
import edu.kit.ipd.sdq.kamp.core.ArchitectureModelFactoryFacade;
import edu.kit.ipd.sdq.kamp.core.ArchitectureModelLookup;
import edu.kit.ipd.sdq.kamp.core.ArchitectureVersion;
import edu.kit.ipd.sdq.kamp.core.ArchitectureVersionPersistency;
import edu.kit.ipd.sdq.kamp.core.BasicActivity;
import edu.kit.ipd.sdq.kamp.core.DifferenceCalculation;

public class DifferenceDetectionTest {

	ArchitectureVersion baseArchitectureVersion;
	ArchitectureVersion targetArchitectureVersion;
	
	@Before
	public void setUp() throws Exception {
		
		baseArchitectureVersion = setupBasePCMModel("basemodel");
		
		ArchitectureVersionPersistency.save(baseArchitectureVersion.getName(), baseArchitectureVersion);
		
		targetArchitectureVersion = saveAsAndReload(baseArchitectureVersion, "targetmodel");
				
		// Test validiationExceptions
		
		// Test 
	}
	
	private ArchitectureVersion saveAsAndReload(
			ArchitectureVersion saveVersion, String targetname) {
		try {
			ArchitectureVersionPersistency.save(targetname, saveVersion);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		ArchitectureVersion loadVersion = null;
		try {
			loadVersion = ArchitectureVersionPersistency.load(targetname, targetname);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}		
		
		return loadVersion;
	}

	@After
	public void tearDown() {
		baseArchitectureVersion.delete();
		targetArchitectureVersion.delete();
		baseArchitectureVersion = null;
		targetArchitectureVersion = null;
	}

	private static ArchitectureVersion setupBasePCMModel(String name) {
		ArchitectureVersion baseversion = ArchitectureModelFactoryFacade.createEmptyModel(name);

		BasicComponent client = ArchitectureModelFactoryFacade.createBasicComponent(baseversion, "Client");
		BasicComponent server = ArchitectureModelFactoryFacade.createBasicComponent(baseversion, "Server");
		BasicComponent database = ArchitectureModelFactoryFacade.createBasicComponent(baseversion, "Database");
		
		OperationInterface clientInterface = ArchitectureModelFactoryFacade.createInterface(baseversion, "ClientInterface");
		OperationInterface serverInterface = ArchitectureModelFactoryFacade.createInterface(baseversion, "ServerInterface");
		OperationInterface databaseInterface = ArchitectureModelFactoryFacade.createInterface(baseversion, "DatabaseInterface");
		
		ArchitectureModelFactoryFacade.createProvidedRole(client, clientInterface);
		ArchitectureModelFactoryFacade.createRequiredRole(client, serverInterface);

		ArchitectureModelFactoryFacade.createProvidedRole(server, serverInterface);
		ArchitectureModelFactoryFacade.createRequiredRole(server, databaseInterface);

		ArchitectureModelFactoryFacade.createProvidedRole(database, databaseInterface);

		ArchitectureModelFactoryFacade.createAssemblyContext(client, baseversion);
		ArchitectureModelFactoryFacade.createAssemblyContext(server, baseversion);
		ArchitectureModelFactoryFacade.createAssemblyContext(database, baseversion);
		
		//ArchitectureModelFactoryFacade.createAssemblyConnector(server, client, baseversion);
		
		return baseversion;
	}
	
	@Test
	public void testDeepClone() {
		assertTrue("TargetRepository is null", targetArchitectureVersion.getRepository() != null);
		System.out.println("baseModelId:  "+baseArchitectureVersion.getRepository().getId());
		System.out.println("targetModelId:"+targetArchitectureVersion.getRepository().getId());
		assertTrue("Repository-ID-Check failed", baseArchitectureVersion.getRepository().getId().equals(targetArchitectureVersion.getRepository().getId()));
		assertTrue("Number of Components different", baseArchitectureVersion.getRepository().getComponents__Repository().size()==targetArchitectureVersion.getRepository().getComponents__Repository().size());

		for (int i=0; i < baseArchitectureVersion.getRepository().getComponents__Repository().size(); i++) {
			RepositoryComponent sourceComponent = baseArchitectureVersion.getRepository().getComponents__Repository().get(i);
			RepositoryComponent targetComponent = targetArchitectureVersion.getRepository().getComponents__Repository().get(i);
			
			assertTrue("Components are identical", sourceComponent!=targetComponent);
			assertTrue("Component-ID-Check failed", sourceComponent.getId().equals(targetComponent.getId()));
			assertTrue("Component: number of provided roles different", sourceComponent.getProvidedRoles_InterfaceProvidingEntity().size()==targetComponent.getProvidedRoles_InterfaceProvidingEntity().size());
			assertTrue("Component: number of required roles different", sourceComponent.getRequiredRoles_InterfaceRequiringEntity().size()==targetComponent.getRequiredRoles_InterfaceRequiringEntity().size());
		}
		
		assertTrue("Interface for Provided Role is null", ((OperationProvidedRole)targetArchitectureVersion.getRepository().getComponents__Repository().get(0).getProvidedRoles_InterfaceProvidingEntity().get(0)).getProvidedInterface__OperationProvidedRole()!=null);
		
		assertTrue("Number of Interfaces in Repository different", baseArchitectureVersion.getRepository().getInterfaces__Repository().size()==targetArchitectureVersion.getRepository().getInterfaces__Repository().size());
		assertTrue("Number of Datatypes in Repository different", baseArchitectureVersion.getRepository().getDataTypes__Repository().size()==targetArchitectureVersion.getRepository().getDataTypes__Repository().size());
	}
	
	@Test 
	public void testEMFCompareDiffUsage() {
		
		// Do Modifications
		BasicComponent testComponent = ArchitectureModelFactoryFacade.createBasicComponent(targetArchitectureVersion, "AddedTestComponent");

		OperationInterface testInterface = ArchitectureModelFactoryFacade.createInterface(targetArchitectureVersion, "AddedTestInterface");
		
		ArchitectureModelFactoryFacade.createProvidedRole(testComponent, testInterface);

		// Step 1: Matching 
		
		// explanation: MatchService.doMatch(local, latestFromRepository, null=default options);
		MatchModel matchModel = null;
		try {
			matchModel = MatchService.doMatch(targetArchitectureVersion.getRepository(), baseArchitectureVersion.getRepository(), null);
		} catch (InterruptedException e) {
			fail("Model Matching interrupted, "+e.getMessage());
		}

		assertTrue("MatchModel null", matchModel!=null);
		
		if (matchModel != null) {
			DiffModel diff = DiffService.doDiff(matchModel);
			
			assertTrue("DiffModel null", diff!=null);
			assertTrue("DiffModel is empty (no contents)", !diff.eContents().isEmpty());

//			for (TreeIterator<EObject> iterator = diff.eAllContents(); iterator.hasNext();) {
//				EObject el = iterator.next();
//				System.out.println(el.eClass().getName()+": "+ el.toString());
//	        }
			
		}
		
	}
	
	// Die folgenden Tests sollen atomare Modellvernderungen testen
	
	@Test 
	public void testDifferenceCalculationOnRepository() {
//		// add component to repository
//		PCMFactory.createBasicComponent(targetArchitectureVersion.getRepository(), "AddedTestComponent");
//		boolean addedBasicComponentFound = false;
//
//		// add interface to repository
//		PCMFactory.createInterface(targetArchitectureVersion.getRepository(), "AddedTestInterface");
//		boolean addedInterfaceFound = false;

//		// remove component from repository
//		EcoreUtil.delete(targetArchitectureVersion.getRepository().getComponents__Repository().get(0));

//		// remove interface from repository
//		EcoreUtil.delete(targetArchitectureVersion.getRepository().getInterfaces__Repository().get(0));
//		EcoreUtil.delete(targetArchitectureVersion.getRepository().getInterfaces__Repository().get(0));
		
//		// add interface to repository
//		OperationInterface testInterface = PCMFactory.createInterface(targetArchitectureVersion.getRepository(), "AddedTestInterface");
//		
//		// add provided role
//		PCMFactory.createProvidedRole((BasicComponent)targetArchitectureVersion.getRepository().getComponents__Repository().get(0), testInterface);
//		// add required role
//		PCMFactory.createRequiredRole((BasicComponent)targetArchitectureVersion.getRepository().getComponents__Repository().get(0), testInterface);

		// delete provided role
		//EcoreUtil.delete(targetArchitectureVersion.getRepository().getComponents__Repository().get(0).getProvidedRoles_InterfaceProvidingEntity().get(0));

		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());
		
		System.out.println("elements from tree-iterator:");
		for (TreeIterator<EObject> iterator = diff.eAllContents(); iterator.hasNext();) {
			EObject el = iterator.next();
			System.out.println(el.eClass().getName()+": "+ el.toString());
    	}
		
		System.out.println("elements from differences-list:");
		for (DiffElement diffElement : diff.getDifferences()) {
			System.out.println(">>> "+diffElement.eClass().getName());
			System.out.println(diffElement.toString());
			System.out.println("kind: "+ diffElement.getKind().toString());
			System.out.println("has subdiffs: "+ !diffElement.getSubDiffElements().isEmpty());
		
			if (diffElement instanceof ModelElementChangeLeftTarget) {
				ModelElementChangeLeftTarget element = (ModelElementChangeLeftTarget) diffElement;
				System.out.println("Left Element: "+ element.getLeftElement());
				System.out.println("Right Parent: "+ element.getRightParent());
				
				if (element.getLeftElement() instanceof BasicComponent) {
					BasicComponent basicComponent = (BasicComponent) element.getLeftElement();
					System.out.println("Action: Basic Component "+basicComponent+" Added");
				} else if (element.getLeftElement() instanceof OperationInterface) {
					OperationInterface operationInterface = (OperationInterface) element.getLeftElement();
					System.out.println("Action: Interface "+operationInterface+" Added");
				} else if (element.getLeftElement() instanceof OperationRequiredRole) {
					OperationRequiredRole requiredRole = (OperationRequiredRole) element.getLeftElement();
					System.out.println("Action: OperationRequiredRole "+requiredRole+" Added");
				} else if (element.getLeftElement() instanceof OperationProvidedRole) {
					OperationProvidedRole providedRole = (OperationProvidedRole) element.getLeftElement();
					System.out.println("Action: OperationProvidedRole "+providedRole+" Added");
				} else {
					System.out.println("not matched");
				}
			} else if (diffElement instanceof ModelElementChangeRightTarget) {
				ModelElementChangeRightTarget element = (ModelElementChangeRightTarget) diffElement;
				System.out.println("Right Element: "+ element.getRightElement());
				System.out.println("Left Parent: "+ element.getLeftParent());

				if (element.getRightElement() instanceof BasicComponent) {
					BasicComponent basicComponent = (BasicComponent) element.getRightElement();
					System.out.println("Action: Basic Component "+basicComponent+" Deleted");
				} else if (element.getRightElement() instanceof OperationInterface) {
					OperationInterface operationInterface = (OperationInterface) element.getRightElement();
					System.out.println("Action: Interface "+operationInterface+" Deleted");
				} else {
					System.out.println("not matched");
				}
			} else if (diffElement instanceof UpdateReference) {
				UpdateReference element = (UpdateReference) diffElement;
				System.out.println("Left Element: "+element.getLeftElement());
				System.out.println("Left Target: "+element.getLeftTarget());
				System.out.println("Right Element: "+element.getRightElement());
				System.out.println("Right Target: "+element.getRightTarget());
				if ((element.getKind() == DifferenceKind.CHANGE)
						&& (element.getRightTarget() == null)
						&& (element.getLeftElement() instanceof ProvidedRole)
						&& (element.getLeftTarget() instanceof OperationInterface)) {
					System.out.println("Action: Interface for Provided Role deleted.");
				} else if ((element.getKind() == DifferenceKind.CHANGE)
						&& (element.getRightTarget() == null)
						&& (element.getLeftElement() instanceof RequiredRole)
						&& (element.getLeftTarget() instanceof OperationInterface)) {
					System.out.println("Action: Interface for Required Role deleted.");
				} else {
					System.out.println("not matched");
				}
			} else {
				System.out.println("not matched");
			}
			System.out.println("----");
		}
		
	}
	
	
	
	
	@Test 
	public void testDifferenceDetectionRules_AddedBasicComponent() {
		// add component to repository
		ArchitectureModelFactoryFacade.createBasicComponent(targetArchitectureVersion, "AddedTestComponent");
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());
		
		List<DiffElement> diffElements = DifferenceCalculation.foundAddedBasicComponent(diff);
		
		assertTrue("Added Basic Component Not Found", diffElements.size()==1);

		EObject component = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
				
		assertTrue("Added Basic Component Not Retrieved Properly", component!=null && component instanceof BasicComponent);
	}
	
	@Test 
	public void testDifferenceDetectionRules_RemovedBasicComponent() {
		// remove component from repository
		EcoreUtil.delete(targetArchitectureVersion.getRepository().getComponents__Repository().get(0));
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundDeletedBasicComponent(diff);
		
		assertTrue("Deleted Basic Component Not Found", diffElements.size()==1);

		EObject component = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Deleted Basic Component Not Retrieved Properly", component!=null && component instanceof BasicComponent);

	}

	@Test 
	public void testDifferenceDetectionRules_AddedInterface() {
		// add interface to repository
		ArchitectureModelFactoryFacade.createInterface(targetArchitectureVersion, "AddedTestInterface");
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundAddedInterface(diff);

		assertTrue("Added Interface Not Found", diffElements.size()==1);

		EObject interf = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Added Interface Not Retrieved Properly", interf!=null && interf instanceof OperationInterface);

	}

	@Test 
	public void testDifferenceDetectionRules_RemovedInterface() {
		// remove interface
		EcoreUtil.delete(targetArchitectureVersion.getRepository().getInterfaces__Repository().get(0));
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundDeletedInterface(diff);

		assertTrue("Deleted Interface Not Found", diffElements.size()==1);

		EObject interf = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Deleted Interface Not Retrieved Properly", interf!=null && interf instanceof OperationInterface);
	}
	
	@Test 
	public void testDifferenceDetectionRules_RemovedInterfaceForProvidedRole() {
		// remove interface
		EcoreUtil.delete(targetArchitectureVersion.getRepository().getInterfaces__Repository().get(0));
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundDeletedInterfaceForProvidedRole(diff);

		assertTrue("Deleted Interface For ProvidedRole Not Found", diffElements.size()==1);

		EObject role = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Provided Role with Deleted Interface Not Retrieved Properly", role!=null && role instanceof ProvidedRole);
	}
	
	@Test 
	public void testDifferenceDetectionRules_RemovedInterfaceForRequiredRole() {
		// remove interface
		EcoreUtil.delete(targetArchitectureVersion.getRepository().getInterfaces__Repository().get(1));
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundDeletedInterfaceForRequiredRole(diff);

		assertTrue("Deleted Interface For RequiredRole Not Found", diffElements.size()==1);

		EObject role = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Required Role with Deleted Interface Not Retrieved Properly", role!=null && role instanceof RequiredRole);
	}
	

	@Test 
	public void testDifferenceDetectionRules_AddProvidedRole() {
		// add interface to repository
		OperationInterface testInterface = ArchitectureModelFactoryFacade.createInterface(targetArchitectureVersion, "AddedTestInterface");
		// add provided role
		ArchitectureModelFactoryFacade.createProvidedRole((BasicComponent)targetArchitectureVersion.getRepository().getComponents__Repository().get(0), testInterface);		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundAddedProvidedRole(diff);
		
		assertTrue("Added Provided Role not found", diffElements.size()==1);

		EObject role = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Added Provided Role not Retrieved Properly", role!=null && role instanceof ProvidedRole);

	}

	@Test 
	public void testDifferenceDetectionRules_AddRequiredRole() {
		// add interface to repository
		OperationInterface testInterface = ArchitectureModelFactoryFacade.createInterface(targetArchitectureVersion, "AddedTestInterface");
		// add required role
		ArchitectureModelFactoryFacade.createRequiredRole((BasicComponent)targetArchitectureVersion.getRepository().getComponents__Repository().get(0), testInterface);	
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundAddedRequiredRole(diff);
		
		assertTrue("Added Required Role not found", diffElements.size()==1);

		EObject role = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Added Required Role not Retrieved Properly", role!=null && role instanceof RequiredRole);
	}
	
	@Test 
	public void testDifferenceDetectionRules_DeleteProvidedRole() {
		EcoreUtil.delete(targetArchitectureVersion.getRepository().getComponents__Repository().get(0).getProvidedRoles_InterfaceProvidingEntity().get(0));
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundDeletedProvidedRole(diff);

		assertTrue("Deleted Provided Role not found", diffElements.size()==1);

		EObject role = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Deleted Provided Role not Retrieved Properly", role!=null && role instanceof ProvidedRole);

	}

	@Test 
	public void testDifferenceDetectionRules_DeleteRequiredRole() {
		EcoreUtil.delete(targetArchitectureVersion.getRepository().getComponents__Repository().get(0).getRequiredRoles_InterfaceRequiringEntity().get(0));
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());

		List<DiffElement> diffElements = DifferenceCalculation.foundDeletedRequiredRole(diff);

		assertTrue("Deleted Required Role not found", diffElements.size()==1);

		EObject role = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
		
		assertTrue("Deleted Provided Role not Retrieved Properly", role!=null && role instanceof RequiredRole);

	}
	
	@Test 
	public void testWorkplanGeneration_AddBasicComponent() {
		// add component to repository
		ArchitectureModelFactoryFacade.createBasicComponent(targetArchitectureVersion, "AddedTestComponent");
		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getRepository(), targetArchitectureVersion.getRepository());
		
		List<Activity> workplan = DifferenceCalculation.deriveBaseWorkPlan(diff);

		assertTrue("Added Basic Component Not Found", (!workplan.isEmpty())&&(workplan.get(0).getBasicActivity()==BasicActivity.ADD)&&(workplan.get(0).getElementType()==ActivityElementType.COMPONENT));	
	}
	
	@Test 
	public void testDifferenceCalculationOnSystem() {
		// assemblycontexts added
		
		RepositoryComponent client = ArchitectureModelLookup.lookUpComponentByName(targetArchitectureVersion, "Client");
		RepositoryComponent server = ArchitectureModelLookup.lookUpComponentByName(targetArchitectureVersion, "Server");
		
		// assembly connector added
		ArchitectureModelFactoryFacade.createAssemblyConnector(client, server, targetArchitectureVersion);
		
		// assembly connector removed

		
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getSystem(), targetArchitectureVersion.getSystem());

		System.out.println("elements from tree-iterator:");
		for (TreeIterator<EObject> iterator = diff.eAllContents(); iterator.hasNext();) {
			EObject el = iterator.next();
			System.out.println(el.eClass().getName()+": "+ el.toString());
    	}
		
		System.out.println("elements from differences-list:");
		for (DiffElement diffElement : diff.getDifferences()) {
			System.out.println(">>> "+diffElement.eClass().getName());
			System.out.println(diffElement.toString());
			System.out.println("kind: "+ diffElement.getKind().toString());
			System.out.println("has subdiffs: "+ !diffElement.getSubDiffElements().isEmpty());
		}
	}
	
	@Test 
	public void testDifferenceDetectionRules_System_AssemblyConnectorAdded() {
		RepositoryComponent client = ArchitectureModelLookup.lookUpComponentByName(targetArchitectureVersion, "Client");
		RepositoryComponent server = ArchitectureModelLookup.lookUpComponentByName(targetArchitectureVersion, "Server");
		
		// assembly connector added
		ArchitectureModelFactoryFacade.createAssemblyConnector(client, server, targetArchitectureVersion);
	
		DiffModel diff = DifferenceCalculation.calculateDiffModel(baseArchitectureVersion.getSystem(), targetArchitectureVersion.getSystem());
	
		List<DiffElement> diffElements = DifferenceCalculation.foundAddedAssemblyConnector(diff);
		
		assertTrue("Added Assembly Connector Not Found", diffElements.size()==1);

		EObject connector = DifferenceCalculation.retrieveArchitectureElement(diffElements.get(0));
				
		assertTrue("Added Assembly Connector Not Retrieved Properly", connector!=null && connector instanceof AssemblyConnector);
	}
	
	@Test 
	public void testArchitectureModelLookup_lookUpComponentByName() {
		String testcomponentName = "LookUpComponent";
		
		BasicComponent testComponent = ArchitectureModelFactoryFacade.createBasicComponent(baseArchitectureVersion, testcomponentName);
		BasicComponent foundComponent = (BasicComponent) ArchitectureModelLookup.lookUpComponentByName(baseArchitectureVersion, testcomponentName);
		
		assertTrue("Component not found", foundComponent!=null);
		assertTrue("Wrong Component found", foundComponent==testComponent);

		BasicComponent errorComponent = (BasicComponent) ArchitectureModelLookup.lookUpComponentByName(baseArchitectureVersion, null);
		assertTrue("LookUp of missing component not resulted in null", errorComponent==null);
	}	
	
}
